'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

var _timeoutError = require('../timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud CopyFactory configuration API (trade copying configuration API) client (see
 * https://metaapi.cloud/docs/copyfactory/)
 */
class ConfigurationClient extends _metaApi2.default {

  /**
   * Constructs CopyFactory configuration API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */
  constructor(httpClient, token, domain = 'agiliumtrade.agiliumtrade.ai') {
    super(httpClient, token, domain);
    this._host = `https://copyfactory-application-history-master-v1.${domain}`;
  }

  /**
   * Strategy id
   * @typedef {Object} StrategyId
   * @property {String} id strategy id
   */

  /**
   * Retrieves new unused strategy id. Method is accessible only with API access token. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/generateStrategyId/
   * @return {Promise<StrategyId>} promise resolving with strategy id generated
   */
  generateStrategyId() {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('generateStrategyId');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/unused-strategy-id`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Generates random account id
   * @return {String} account id
   */
  generateAccountId() {
    return _randomstring2.default.generate(64);
  }

  /**
   * CopyFactory strategy subscriptions
   * @typedef {Object} CopyFactoryStrategySubscription
   * @property {String} strategyId id of the strategy to subscribe to
   * @property {Number} [multiplier] subscription multiplier, default is 1x
   * @property {Boolean} [skipPendingOrders] flag indicating that pending orders should not be copied. Default
   * is to copy pending orders
   * @property {String} [closeOnly] setting wich instructs the application not to open new positions. by-symbol
   * means that it is still allowed to open new positions with a symbol equal to the symbol of an existing strategy
   * position (can be used to gracefuly exit strategies trading in netting mode or placing a series of related trades
   * per symbol). immediately means to close all positions immediately. One of 'by-position', 'by-symbol', 'immediately'
   * @property {Number} [maxTradeRisk] max risk per trade, expressed as a fraction of 1. If trade has a SL, the
   * trade size will be adjusted to match the risk limit. If not, the trade SL will be applied according to the risk
   * limit
   * @property {Boolean} [reverse] flag indicating that the strategy should be copied in a reverse direction
   * @property {String} [reduceCorrelations] setting indicating whether to enable automatic trade
   * correlation reduction. Possible settings are not specified (disable correlation risk restrictions),
   * by-strategy (limit correlations for the strategy) or by-account (limit correlations for the account)
   * @property {CopyFactoryStrategyStopOutSettings} [stopOutRisk] stop out setting. All trading will be terminated
   * and positions closed once equity drawdown reaches this value
   * @property {CopyFactoryStrategySymbolFilter} [symbolFilter] symbol filter which can be used to copy only specific
   * symbols or exclude some symbols from copying
   * @property {CopyFactoryStrategyNewsFilter} [newsFilter] news risk filter configuration
   * @property {Array<CopyFactoryStrategyRiskLimit>} [riskLimits] strategy risk limits. You can configure trading to be
   * stopped once total drawdown generated during specific period is exceeded. Can be specified either for balance or
   * equity drawdown
   * @property {CopyFactoryStrategyMaxStopLoss} [maxStopLoss] stop loss value restriction
   * @property {Number} [maxLeverage] setting indicating maximum leverage allowed when opening a new positions.
   * Any trade which results in a higher leverage will be discarded
   * @property {Array<CopyFactoryStrategySymbolMapping>} [symbolMapping] defines how symbol name should be changed when
   * trading (e.g. when broker uses symbol names with unusual suffixes). By default this setting is disabled and the
   * trades are copied using signal source symbol name
   * @property {CopyFactoryStrategyTradeSizeScaling} [tradeSizeScaling] Trade size scaling settings. By default the
   * trade size on strategy subscriber side will be scaled according to balance to preserve risk.
   * @property {boolean} [copyStopLoss] flag indicating whether stop loss should be copied. Default is to copy stop
   * loss.
   * @property {boolean} [copyTakeProfit] flag indicating whether take profit should be copied. Default is to copy take
   * profit.
   * @property {number} [minTradeVolume] Minimum trade volume to copy. Trade signals with a smaller volume will not be
   * copied
   * @property {number} [maxTradeVolume] Maximum trade volume to copy. Trade signals with a larger volume will be copied
   * with maximum volume instead
   * @property {boolean} [removed] flag indicating that the subscription was scheduled for removal once all subscription
   * positions will be closed
   */

  /**
   * CopyFactory strategy trade size scaling settings
   * @typedef {Object} CopyFactoryStrategyTradeSizeScaling
   * @property {string} mode If set to balance, the trade size on strategy subscriber will be scaled according to
   * balance to preserve risk. If value is none, then trade size will be preserved irregardless of the subscriber
   * balance. If value is contractSize, then trade size will be scaled according to contract size. If fixedVolume is
   * set, then trade will be copied with a fixed volume of traceVolume setting. If fixedRisk is set, then each trade
   * will be copied with a trade volume set to risk specific fraction of balance as configured by riskFraction setting.
   * Note, that in fixedRisk mode trades without a SL are not copied. Default is balance. Allowed values: none,
   * contractSize, balance, fixedVolume, fixedRisk
   * @property {number} [tradeVolume] Fixed trade volume for use with fixedVolume trade size scaling mode
   * @property {number} [riskFraction] Fixed risk fraction for use with fixedRisk trade size scaling mode
   */

  /**
   * CopyFactory strategy stopout settings
   * @typedef {Object} CopyFactoryStrategyStopOutSettings
   * @property {Number} [absoluteValue] value of the stop out risk, measured in account currency
   * @property {Number} [relativeValue] value of the stop out risk, expressed as a fraction of 1
   * @property {Date|string|moment.Moment} [startTime] the time to start risk calculation from. All previous trades will be ignored. You can
   * use it to reset the risk counter after a stopout event
   */

  /**
   * CopyFactory symbol filter
   * @typedef {Object} CopyFactoryStrategySymbolFilter
   * @property {Array<String>} included list of symbols copied. Leave the value empty to copy all symbols
   * @property {Array<String>} excluded list of symbols excluded from copying. Leave the value empty to copy all symbols
   */

  /**
   * CopyFactory news risk filter
   * @typedef {Object} CopyFactoryStrategyNewsFilter
   * @property {CopyFactoryStrategyBreakingNewsFilter} [breakingNewsFilter] breaking news filter
   * @property {CopyFactoryStrategyCalendarNewsFilter} [calendarNewsFilter] calendar news filter
   */

  /**
   * CopyFactory breaking news risk filter
   * @typedef {Object} CopyFactoryStrategyBreakingNewsFilter
   * @property {Array<String>} priorities list of breaking news priorities to stop trading on, leave empty to disable
   * breaking news filter. One of high, medium, low.
   * @property {Number} [closePositionTimeGapInMinutes] time interval specifying when to force close an already
   * open position after breaking news. Default value is 60 minutes
   * @property {Number} [openPositionFollowingTimeGapInMinutes] time interval specifying when it is allowed to
   * open position after calendar news. Default value is 60 minutes
   */

  /**
   * CopyFactory calendar new filter
   * @typedef {Object} CopyFactoryStrategyCalendarNewsFilter
   * @property {Array<String>} priorities list of calendar news priorities to stop trading on, leave empty to disable
   * calendar news filter. One of election, high, medium, low.
   * @property {Number} [closePositionTimeGapInMinutes] time interval specifying when to force close an already
   * open position before calendar news. Default value is 60 minutes
   * @property {Number} [openPositionPrecedingTimeGapInMinutes] time interval specifying when it is still
   * allowed to open position before calendar news. Default value is 120 minutes
   * @property {Number} [openPositionFollowingTimeGapInMinutes] time interval specifying when it is allowed to
   * open position after calendar news. Default value is 60 minutes
   */

  /**
   * CopyFactory risk limit filter
   * @typedef {Object} CopyFactoryStrategyRiskLimit
   * @property {String} type restriction type. One of daily, monthly, or yearly
   * @property {String} applyTo account metric to apply limit to. One of balance, equity
   * @property {Number} [maxAbsoluteRisk] max drawdown allowed, measured in account currency
   * @property {Number} [maxRelativeRisk] max drawdown allowed, measured in account currency
   * @property {Boolean} closePositions whether to force close positions when the risk is reached. If value is false
   * then only the new trades will be halted, but existing ones will not be closed
   * @property {Date|string|moment.Moment} [startTime] time to start risk tracking from. All previous trades will be ignored. You
   * can use this value to reset the filter after stopout event
   */

  /**
   * CopyFactory strategy max stop loss settings
   * @typedef {Object} CopyFactoryStrategyMaxStopLoss
   * @property {Number} value maximum SL value
   * @property {String} units SL units. Only pips value is supported at this point
   */

  /**
   * CopyFactory strategy symbol mapping
   * @typedef {Object} CopyFactoryStrategySymbolMapping
   * @property {String} from symbol name to convert from
   * @property {String} to symbol name to convert to
   */

  /**
   * CopyFactory account update
   * @typedef {Object} CopyFactorySubscriberUpdate
   * @property {String} name account human-readable name
   * @property {Number} [reservedMarginFraction] fraction of reserved margin to reduce a risk of margin call.
   * Default is to reserve no margin. We recommend using maxLeverage setting instead. Specified as a fraction of balance
   * thus the value is usually greater than 1
   * @property {Array<String>} [phoneNumbers] phone numbers to send sms notifications to. Leave empty to
   * receive no sms notifications
   * @property {Number} [minTradeAmount] value of minimal trade size allowed, expressed in amount of account
   * currency. Can be useful if your broker charges a fixed fee per transaction so that you can skip small trades with
   * high broker commission rates. Default is 0
   * @property {String} [closeOnly] setting wich instructs the application not to open new positions. by-symbol
   * means that it is still allowed to open new positions with a symbol equal to the symbol of an existing strategy
   * position (can be used to gracefuly exit strategies trading in netting mode or placing a series of related trades
   * per symbol). immediately means to close all positions immediately. One of 'by-position', 'by-symbol', 'immediately'
   * @property {CopyFactoryStrategyStopOutSettings} [stopOutRisk] stop out setting. All trading will
   * be terminated and positions closed once equity drawdown reaches this value
   * @property {Array<CopyFactoryStrategyRiskLimit>} [riskLimits] account risk limits. You can configure trading to be
   * stopped once total drawdown generated during specific period is exceeded. Can be specified either for balance or
   * equity drawdown
   * @property {Number} [maxLeverage] setting indicating maxumum leverage allowed when opening a new positions.
   * Any trade which results in a higher leverage will be discarded.
   * @property {boolean} [copyStopLoss] flag indicating whether stop loss should be copied. Default is to copy stop
   * loss.
   * @property {boolean} [copyTakeProfit] flag indicating whether take profit should be copied. Default is to copy take
   * profit.
   * @property {number} [minTradeVolume] Minimum trade volume to copy. Trade signals with a smaller volume will not be
   * copied
   * @property {number} [maxTradeVolume] Maximum trade volume to copy. Trade signals with a larger volume will be copied
   * with maximum volume instead
   * @property {Array<CopyFactoryStrategySubscription>} subscriptions strategy subscriptions
   */

  /**
   * CopyFactory subscriber model
   * @typedef {CopyFactorySubscriberUpdate} CopyFactorySubscriber
   * @property {String} _id id of the MetaApi account to copy trades to 
   */

  /**
   * CopyFactory provider strategy
   * @typedef {CopyFactoryStrategyUpdate} CopyFactoryStrategy
   * @property {String} _id unique strategy id
   * @property {Number} platformCommissionRate commission rate the platform charges for strategy copying, applied to
   * commissions charged by provider. This commission applies only to accounts not managed directly by provider. Should
   * be fraction of 1
   * @property {String} [closeOnRemovalMode] position close mode on strategy or subscription removal. Preserve means
   * that positions will not be closed and will not be managed by CopyFactory. close-gracefully-by-position means
   * that positions will continue to be managed by CopyFactory, but only close signals will be copied.
   * close-gracefully-by-symbol means that positions will continue to be managed by CopyFactory, but only close
   * signals will be copied or signals to open positions for the symbols which already have positions opened.
   * close-immediately means that all positions will be closed immediately. Default is close-immediately.
   * This field can be changed via remove potfolio member API only, one of preserve, close-gracefully-by-position,
   * close-gracefully-by-symbol, close-immediately
   */

  /**
   * CopyFactory strategy commission scheme
   * @typedef {Object} CopyFactoryStrategyCommissionScheme
   * @property {String} type commission type. One of flat-fee, lots-traded, lots-won, amount-traded, amount-won,
   * high-water-mark
   * @property {String} billingPeriod billing period. One of week, month, quarter
   * @property {Number} commissionRate commission rate. Should be greater than or equal to zero if commission type is
   * flat-fee, lots-traded or lots-won, should be greater than or equal to zero and less than or equal to 1 if
   * commission type is amount-traded, amount-won, high-water-mark.
   */

  /**
   * CopyFactory strategy magic filter
   * @typedef {Object} CopyFactoryStrategyMagicFilter
   * @property {Array<String>} included list of magics (expert ids) or magic ranges copied. Leave the value empty to
   * copy all magics
   * @property {Array<String>} excluded list of magics (expert ids) or magic ranges excluded from copying. Leave the
   * value empty to copy all magics
   */

  /**
   * CopyFactory strategy time settings
   * @typedef {Object} CopyFactoryStrategyTimeSettings
   * @property {Number} [lifetimeInHours] position lifetime. Default is to keep positions open up to 90 days
   * @property {Number} [openingIntervalInMinutes] time interval to copy new positions. Default is to let 1
   * minute for the position to get copied. If position were not copied during this time, the copying will not be
   * retried anymore.
   */

  /**
   * CopyFactory strategy equity curve filter
   * @typedef {Object} CopyFactoryStrategyEquityCurveFilter
   * @property {Number} period moving average period, must be greater or equal to 1
   * @property {String} timeframe moving average timeframe, a positive integer followed by time unit, e.g. 2h.
   * Allowed units are s, m, h, d and w.
   */

  /**
   * Retrieves CopyFactory copy trading strategies. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getStrategies/
   * @param {Boolean} [includeRemoved] flag instructing to include removed strategies in results
   * @param {Number} [limit] pagination limit
   * @param {Number} [offset] copy trading strategy id
   * @return {Promise<Array<CopyFactoryStrategy>>} promise resolving with CopyFactory strategies found
   */
  getStrategies(includeRemoved, limit, offset) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getStrategies');
    }
    let qs = {};
    if (includeRemoved !== undefined) {
      qs.includeRemoved = includeRemoved;
    }
    if (limit !== undefined) {
      qs.limit = limit;
    }
    if (offset !== undefined) {
      qs.offset = offset;
    }
    const opts = {
      url: `${this._host}/users/current/configuration/strategies`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      qs,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves CopyFactory copy trading strategy by id. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getStrategy/
   * @param {string} strategy id trading strategy id
   * @return {Promise<CopyFactoryStrategy>} promise resolving with CopyFactory strategy found
   */
  getStrategy(strategyId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/strategies/${strategyId}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * CopyFactory strategy drawdown filter
   * @typedef {Object} CopyFactoryStrategyDrawdownFilter
   * @property {Number} maxDrawdown maximum drawdown value after which action is executed. Drawdown should
   * be configured as a fraction of 1, i.e. 0.15 means 15% drawdown value
   * @property {String} action action to take when drawdown exceeds maxDrawdown value. include means the trading
   * signal will be transmitted only if dd is greater than maxDrawdown value. exclude means the trading signal 
   * will be transmitted only if dd is less than maxDrawdown value
   */

  /**
   * CopyFactory strategy update
   * @typedef {Object} CopyFactoryStrategyUpdate
   * @property {String} name strategy human-readable name
   * @property {String} description longer strategy human-readable description
   * @property {String} accountId id of the MetaApi account providing the strategy
   * @property {Boolean} [skipPendingOrders] flag indicating that pending orders should not be copied.
   * Default is to copy pending orders
   * @property {CopyFactoryStrategyCommissionScheme} [commissionScheme] commission scheme allowed by this strategy
   * @property {Number} [maxTradeRisk] max risk per trade, expressed as a fraction of 1. If trade has a SL, the
   * trade size will be adjusted to match the risk limit. If not, the trade SL will be applied according to the risk
   * limit
   * @property {Boolean} [reverse] flag indicating that the strategy should be copied in a reverse direction
   * @property {String} [reduceCorrelations] setting indicating whether to enable automatic trade
   * correlation reduction. Possible settings are not specified (disable correlation risk restrictions),
   * by-strategy (limit correlations for the strategy) or by-account (limit correlations for the account)
   * @property {CopyFactoryStrategyStopOutSettings} [stopOutRisk] stop out setting. All trading will
   * be terminated and positions closed once equity drawdown reaches this value
   * @property {CopyFactoryStrategySymbolFilter} [symbolFilter] symbol filters which can be used to copy only specific
   * symbols or exclude some symbols from copying
   * @property {CopyFactoryStrategyNewsFilter} [newsFilter] news risk filter configuration
   * @property {Array<CopyFactoryStrategyRiskLimit>} [riskLimits] strategy risk limits. You can configure
   * trading to be stopped once total drawdown generated during specific period is exceeded. Can be specified either for
   * balance or equity drawdown
   * @property {CopyFactoryStrategyMaxStopLoss} [maxStopLoss] stop loss value restriction
   * @property {Number} [maxLeverage] max leverage risk restriction. All trades resulting in a leverage value
   * higher than specified will be skipped
   * @property {Array<CopyFactoryStrategySymbolMapping>} [symbolMapping] defines how symbol name should be changed when
   * trading (e.g. when broker uses symbol names with unusual suffixes). By default this setting is disabled and the
   * trades are copied using signal source symbol name
   * @property {CopyFactoryStrategyTradeSizeScaling} [tradeSizeScaling] Trade size scaling settings. By default the
   * trade size on strategy subscriber side will be scaled according to balance to preserve risk.
   * @property {boolean} [copyStopLoss] flag indicating whether stop loss should be copied. Default is to copy stop
   * loss.
   * @property {boolean} [copyTakeProfit] flag indicating whether take profit should be copied. Default is to copy take
   * profit.
   * @property {number} [minTradeVolume] Minimum trade volume to copy. Trade signals with a smaller volume will not be
   * copied
   * @property {number} [maxTradeVolume] Maximum trade volume to copy. Trade signals with a larger volume will be copied
   * with maximum volume instead
   * @property {CopyFactoryStrategyMagicFilter} [magicFilter] magic (expert id) filter
   * @property {CopyFactoryStrategyEquityCurveFilter} [equityCurveFilter] filter which permits the trades only if account
   * equity is greater than balance moving average
   * @property {CopyFactoryStrategyDrawdownFilter} [drawdownFilter] master account strategy drawdown filter
   * @property {Array<String>} [symbolsTraded] symbols traded by this strategy. Specifying the symbols will improve trade
   * latency on first trades made by strategy. If you do not specify this setting the application will monitor the strategy
   * trades and detect the symbols automatically over time 
   * @property {CopyFactoryStrategyTimeSettings} [timeSettings] settings to manage copying timeframe and position
   * lifetime. Default is to copy position within 1 minute from being opened at source and let the position to live for
   * up to 90 days
   */

  /**
   * Updates a CopyFactory strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updateStrategy/
   * @param {String} strategyId copy trading strategy id
   * @param {CopyFactoryStrategyUpdate} strategy trading strategy update
   * @return {Promise} promise resolving when strategy is updated
   */
  updateStrategy(strategyId, strategy) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updateStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/strategies/${strategyId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      body: strategy,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * CopyFactory close instructions
   * @typedef {Object} CopyFactoryCloseInstructions
   * @property {String} [mode] position close mode on strategy or subscription removal. Preserve means
   * that positions will not be closed and will not be managed by CopyFactory.
   * close-gracefully-by-position means that positions will continue to be managed by CopyFactory,
   * but only close signals will be copied. close-gracefully-by-symbol means that positions will
   * continue to be managed by CopyFactory, but only close signals will be copied or signals to
   * open positions for the symbols which already have positions opened. close-immediately means
   * that all positions will be closed immediately. Default is close-immediately. One of 'preserve',
   * 'close-gracefully-by-position', 'close-gracefully-by-symbol', 'close-immediately'
   * @property {Date} [removeAfter] time to force remove object after. The object will be removed after
   * this time, even if positions are not yet closed fully. Default is current date plus 30 days.
   * Can not be less than 30 days or greater than current date plus 90 days. The setting is ignored
   * when a subscription is being removed
   */

  /**
   * Deletes a CopyFactory strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeStrategy/
   * @param {String} strategyId copy trading strategy id
   * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
   * @return {Promise} promise resolving when strategy is removed
   */
  removeStrategy(strategyId, closeInstructions) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removeStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/strategies/${strategyId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      body: closeInstructions,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Portfolio strategy member
   * @typedef {Object} CopyFactoryPortfolioStrategyMember
   * @property {String} strategyId member strategy id
   * @property {Number} multiplier copying multiplier (weight in the portfolio)
   * @property {Boolean} [skipPendingOrders] flag indicating that pending orders should not be copied.
   * Default is to copy pending orders
   * @property {Number} [maxTradeRisk] max risk per trade, expressed as a fraction of 1. If trade has a SL, the
   * trade size will be adjusted to match the risk limit. If not, the trade SL will be applied according to the risk
   * limit
   * @property {Boolean} [reverse] flag indicating that the strategy should be copied in a reverse direction
   * @property {String} [reduceCorrelations] setting indicating whether to enable automatic trade
   * correlation reduction. Possible settings are not specified (disable correlation risk restrictions),
   * by-strategy (limit correlations for the strategy) or by-account (limit correlations for the account)
   * @property {CopyFactoryStrategyStopOutSettings} [stopOutRisk] stop out setting. All trading will
   * be terminated and positions closed once equity drawdown reaches this value
   * @property {CopyFactoryStrategySymbolFilter} [symbolFilter] symbol filters which can be used to copy only specific
   * symbols or exclude some symbols from copying
   * @property {CopyFactoryStrategyNewsFilter} [newsFilter] news risk filter configuration
   * @property {Array<CopyFactoryStrategyRiskLimit>} [riskLimits] strategy risk limits. You can configure
   * trading to be stopped once total drawdown generated during specific period is exceeded. Can be specified either for
   * balance or equity drawdown
   * @property {CopyFactoryStrategyMaxStopLoss} [maxStopLoss] stop loss value restriction
   * @property {Number} [maxLeverage] max leverage risk restriction. All trades resulting in a leverage value
   * higher than specified will be skipped
   * @property {Array<CopyFactoryStrategySymbolMapping>} [symbolMapping] defines how symbol name should be changed when
   * trading (e.g. when broker uses symbol names with unusual suffixes). By default this setting is disabled and the
   * trades are copied using signal source symbol name
   * @property {CopyFactoryStrategyTradeSizeScaling} [tradeSizeScaling] Trade size scaling settings. By default the
   * trade size on strategy subscriber side will be scaled according to balance to preserve risk.
   * @property {boolean} [copyStopLoss] flag indicating whether stop loss should be copied. Default is to copy stop
   * loss.
   * @property {boolean} [copyTakeProfit] flag indicating whether take profit should be copied. Default is to copy take
   * profit.
   * @property {number} [minTradeVolume] Minimum trade volume to copy. Trade signals with a smaller volume will not be
   * copied
   * @property {number} [maxTradeVolume] Maximum trade volume to copy. Trade signals with a larger volume will be copied
   * with maximum volume instead
   * @property {String} [closeOnRemovalMode] position close mode on strategy or subscription removal. Preserve means
   * that positions will not be closed and will not be managed by CopyFactory. close-gracefully-by-position means
   * that positions will continue to be managed by CopyFactory, but only close signals will be copied.
   * close-gracefully-by-symbol means that positions will continue to be managed by CopyFactory, but only close
   * signals will be copied or signals to open positions for the symbols which already have positions opened.
   * close-immediately means that all positions will be closed immediately. Default is close-immediately.
   * This field can be changed via remove potfolio member API only, one of preserve, close-gracefully-by-position,
   * close-gracefully-by-symbol, close-immediately
   */

  /**
   * Portfolio strategy update
   * @typedef {Object} CopyFactoryPortfolioStrategyUpdate
   * @property {String} name strategy human-readable name
   * @property {String} description longer strategy human-readable description
   * @property {Array<CopyFactoryPortfolioStrategyMember>} members array of portfolio members
   * @property {CopyFactoryStrategyCommissionScheme} [commissionScheme] commission scheme allowed by this strategy. By
   * default monthly billing period with no commission is being used
   * @property {Boolean} [skipPendingOrders] flag indicating that pending orders should not be copied.
   * Default is to copy pending orders
   * @property {Number} [maxTradeRisk] max risk per trade, expressed as a fraction of 1. If trade has a SL, the
   * trade size will be adjusted to match the risk limit. If not, the trade SL will be applied according to the risk
   * limit
   * @property {Boolean} [reverse] flag indicating that the strategy should be copied in a reverse direction
   * @property {String} [reduceCorrelations] setting indicating whether to enable automatic trade
   * correlation reduction. Possible settings are not specified (disable correlation risk restrictions),
   * by-strategy (limit correlations for the strategy) or by-account (limit correlations for the account)
   * @property {CopyFactoryStrategyStopOutSettings} [stopOutRisk] stop out setting. All trading will
   * be terminated and positions closed once equity drawdown reaches this value
   * @property {CopyFactoryStrategySymbolFilter} [symbolFilter] symbol filters which can be used to copy only specific
   * symbols or exclude some symbols from copying
   * @property {CopyFactoryStrategyNewsFilter} [newsFilter] news risk filter configuration
   * @property {Array<CopyFactoryStrategyRiskLimit>} [riskLimits] strategy risk limits. You can configure
   * trading to be stopped once total drawdown generated during specific period is exceeded. Can be specified either for
   * balance or equity drawdown
   * @property {CopyFactoryStrategyMaxStopLoss} [maxStopLoss] stop loss value restriction
   * @property {Number} [maxLeverage] max leverage risk restriction. All trades resulting in a leverage value
   * higher than specified will be skipped
   * @property {Array<CopyFactoryStrategySymbolMapping>} [symbolMapping] defines how symbol name should be changed when
   * trading (e.g. when broker uses symbol names with unusual suffixes). By default this setting is disabled and the
   * trades are copied using signal source symbol name
   * @property {CopyFactoryStrategyTradeSizeScaling} [tradeSizeScaling] Trade size scaling settings. By default the
   * trade size on strategy subscriber side will be scaled according to balance to preserve risk.
   * @property {boolean} [copyStopLoss] flag indicating whether stop loss should be copied. Default is to copy stop
   * loss.
   * @property {boolean} [copyTakeProfit] flag indicating whether take profit should be copied. Default is to copy take
   * profit.
   * @property {number} [minTradeVolume] Minimum trade volume to copy. Trade signals with a smaller volume will not be
   * copied
   * @property {number} [maxTradeVolume] Maximum trade volume to copy. Trade signals with a larger volume will be copied
   * with maximum volume instead
   */

  /**
   * Portfolio strategy, i.e. the strategy which includes a set of other strategies
   * @typedef {CopyFactoryPortfolioStrategyUpdate} CopyFactoryPortfolioStrategy
   * @property {String} _id unique strategy id
   * @property {Number} platformCommissionRate commission rate the platform charges for strategy copying, applied to
   * commissions charged by provider. This commission applies only to accounts not managed directly by provider. Should
   * be fraction of 1
   * @property {String} [closeOnRemovalMode] position close mode on strategy or subscription removal. Preserve means
   * that positions will not be closed and will not be managed by CopyFactory. close-gracefully-by-position means
   * that positions will continue to be managed by CopyFactory, but only close signals will be copied.
   * close-gracefully-by-symbol means that positions will continue to be managed by CopyFactory, but only close
   * signals will be copied or signals to open positions for the symbols which already have positions opened.
   * close-immediately means that all positions will be closed immediately. Default is close-immediately.
   * This field can be changed via remove potfolio member API only, one of preserve, close-gracefully-by-position,
   * close-gracefully-by-symbol, close-immediately
   */

  /**
   * Retrieves CopyFactory copy portfolio strategies. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getPortfolioStrategies/
   * @param {Boolean} [includeRemoved] flag instructing to include removed portfolio strategies in results
   * @param {Number} [limit] pagination limit
   * @param {Number} [offset] copy trading strategy id
   * @return {Promise<Array<CopyFactoryPortfolioStrategy>>} promise resolving with CopyFactory portfolio strategies
   * found
   */
  getPortfolioStrategies(includeRemoved, limit, offset) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getPortfolioStrategies');
    }
    let qs = {};
    if (includeRemoved !== undefined) {
      qs.includeRemoved = includeRemoved;
    }
    if (limit !== undefined) {
      qs.limit = limit;
    }
    if (offset !== undefined) {
      qs.offset = offset;
    }
    const opts = {
      url: `${this._host}/users/current/configuration/portfolio-strategies`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      qs,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves CopyFactory copy portfolio strategy by id. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getPortfolioStrategy/
   * @param {string} portfolioId portfolio strategy id
   * @return {Promise<CopyFactoryPortfolioStrategy>} promise resolving with CopyFactory portfolio strategy found
   */
  getPortfolioStrategy(portfolioId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getPortfolioStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/portfolio-strategies/${portfolioId}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Updates a CopyFactory portfolio strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updatePortfolioStrategy/
   * @param {String} portfolioId copy trading portfolio strategy id
   * @param {CopyFactoryPortfolioStrategyUpdate} portfolio portfolio strategy update
   * @return {Promise} promise resolving when portfolio strategy is updated
   */
  updatePortfolioStrategy(portfolioId, portfolio) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updatePortfolioStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/portfolio-strategies/${portfolioId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      body: portfolio,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes a CopyFactory portfolio strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removePortfolioStrategy/
   * @param {String} portfolioId portfolio strategy id
   * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
   * @return {Promise} promise resolving when portfolio strategy is removed
   */
  removePortfolioStrategy(portfolioId, closeInstructions) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removePortfolioStrategy');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/portfolio-strategies/${portfolioId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      body: closeInstructions,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes a CopyFactory portfolio strategy member. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removePortfolioStrategyMember/
   * @param {String} portfolioId portfolio strategy id
   * @param {String} strategyId id of the strategy to delete member for
   * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
   * @return {Promise} promise resolving when portfolio strategy is removed
   */
  removePortfolioStrategyMember(portfolioId, strategyId, closeInstructions) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removePortfolioStrategyMember');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/portfolio-strategies/${portfolioId}/members/${strategyId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      body: closeInstructions,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Returns CopyFactory subscribers the user has configured. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/history/getSubscribers/
   * @param {Boolean} [includeRemoved] flag instructing to include removed subscribers in results
   * @param {Number} [limit] pagination limit
   * @param {Number} [offset] copy trading strategy id
   * @return {Promise<Array<CopyFactorySubscriber>>} promise resolving with subscribers found
   */
  getSubscribers(includeRemoved, limit, offset) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getSubscribers');
    }
    let qs = {};
    if (includeRemoved !== undefined) {
      qs.includeRemoved = includeRemoved;
    }
    if (limit !== undefined) {
      qs.limit = limit;
    }
    if (offset !== undefined) {
      qs.offset = offset;
    }
    const opts = {
      url: `${this._host}/users/current/configuration/subscribers`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      qs,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Returns CopyFactory subscriber by id. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getSubscriber/
   * @param {String} subscriberId subscriber id
   * @returns {Promise<CopyFactorySubscriber>} promise resolving with subscriber found
   */
  getSubscriber(subscriberId) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getSubscriber');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/subscribers/${subscriberId}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Updates CopyFactory subscriber configuration. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updateSubscriber/
   * @param {String} subscriberId subscriber id
   * @param {CopyFactorySubscriberUpdate} subscriber subscriber update
   * @returns {Promise} promise resolving when subscriber is updated
   */
  updateSubscriber(subscriberId, subscriber) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updateSubscriber');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/subscribers/${subscriberId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      body: subscriber,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes subscriber configuration. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeSubscriber/
   * @param {String} subscriberId subscriber id
   * @param {CopyFactoryCloseInstructions} [closeInstructions] subscriber close instructions
   * @returns {Promise} promise resolving when subscriber is removed
   */
  removeSubscriber(subscriberId, closeInstructions) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removeSubscriber');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/subscribers/${subscriberId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      body: closeInstructions,
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes a subscription of subscriber to a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeSubscription/
   * @param {String} subscriberId subscriber id
   * @param {String} strategyId strategy id
   * @param {CopyFactoryCloseInstructions} [closeInstructions] subscriber close instructions
   * @returns {Promise} promise resolving when subscriber is removed
   */
  removeSubscription(subscriberId, strategyId, closeInstructions) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('removeSubscription');
    }
    const opts = {
      url: `${this._host}/users/current/configuration/subscribers/${subscriberId}/subscriptions/${strategyId}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      body: closeInstructions,
      json: true
    };
    return this._httpClient.request(opts);
  }

}
exports.default = ConfigurationClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uL2xpYi9jbGllbnRzL2NvcHlGYWN0b3J5L2NvbmZpZ3VyYXRpb24uY2xpZW50LmVzNiJdLCJuYW1lcyI6WyJDb25maWd1cmF0aW9uQ2xpZW50IiwiTWV0YUFwaUNsaWVudCIsImNvbnN0cnVjdG9yIiwiaHR0cENsaWVudCIsInRva2VuIiwiZG9tYWluIiwiX2hvc3QiLCJnZW5lcmF0ZVN0cmF0ZWd5SWQiLCJfaXNOb3RKd3RUb2tlbiIsIl9oYW5kbGVOb0FjY2Vzc0Vycm9yIiwib3B0cyIsInVybCIsIm1ldGhvZCIsImhlYWRlcnMiLCJfdG9rZW4iLCJqc29uIiwiX2h0dHBDbGllbnQiLCJyZXF1ZXN0IiwiZ2VuZXJhdGVBY2NvdW50SWQiLCJyYW5kb21zdHJpbmciLCJnZW5lcmF0ZSIsImdldFN0cmF0ZWdpZXMiLCJpbmNsdWRlUmVtb3ZlZCIsImxpbWl0Iiwib2Zmc2V0IiwicXMiLCJ1bmRlZmluZWQiLCJnZXRTdHJhdGVneSIsInN0cmF0ZWd5SWQiLCJ1cGRhdGVTdHJhdGVneSIsInN0cmF0ZWd5IiwiYm9keSIsInJlbW92ZVN0cmF0ZWd5IiwiY2xvc2VJbnN0cnVjdGlvbnMiLCJnZXRQb3J0Zm9saW9TdHJhdGVnaWVzIiwiZ2V0UG9ydGZvbGlvU3RyYXRlZ3kiLCJwb3J0Zm9saW9JZCIsInVwZGF0ZVBvcnRmb2xpb1N0cmF0ZWd5IiwicG9ydGZvbGlvIiwicmVtb3ZlUG9ydGZvbGlvU3RyYXRlZ3kiLCJyZW1vdmVQb3J0Zm9saW9TdHJhdGVneU1lbWJlciIsImdldFN1YnNjcmliZXJzIiwiZ2V0U3Vic2NyaWJlciIsInN1YnNjcmliZXJJZCIsInVwZGF0ZVN1YnNjcmliZXIiLCJzdWJzY3JpYmVyIiwicmVtb3ZlU3Vic2NyaWJlciIsInJlbW92ZVN1YnNjcmlwdGlvbiJdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztBQUVBOzs7O0FBQ0E7Ozs7QUFDQTs7Ozs7O0FBRUE7Ozs7QUFJZSxNQUFNQSxtQkFBTixTQUFrQ0MsaUJBQWxDLENBQWdEOztBQUU3RDs7Ozs7O0FBTUFDLGNBQVlDLFVBQVosRUFBd0JDLEtBQXhCLEVBQStCQyxTQUFTLDhCQUF4QyxFQUF3RTtBQUN0RSxVQUFNRixVQUFOLEVBQWtCQyxLQUFsQixFQUF5QkMsTUFBekI7QUFDQSxTQUFLQyxLQUFMLEdBQWMscURBQW9ERCxNQUFPLEVBQXpFO0FBQ0Q7O0FBRUQ7Ozs7OztBQU1BOzs7OztBQUtBRSx1QkFBcUI7QUFDbkIsUUFBSSxLQUFLQyxjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixvQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSxpREFEUjtBQUVYTSxjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YQyxZQUFNO0FBTkssS0FBYjtBQVFBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7OztBQUlBUSxzQkFBb0I7QUFDbEIsV0FBT0MsdUJBQWFDLFFBQWIsQ0FBc0IsRUFBdEIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBOENBOzs7Ozs7Ozs7Ozs7OztBQWNBOzs7Ozs7Ozs7QUFTQTs7Ozs7OztBQU9BOzs7Ozs7O0FBT0E7Ozs7Ozs7Ozs7O0FBV0E7Ozs7Ozs7Ozs7Ozs7QUFhQTs7Ozs7Ozs7Ozs7OztBQWFBOzs7Ozs7O0FBT0E7Ozs7Ozs7QUFPQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQWtDQTs7Ozs7O0FBTUE7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBaUJBOzs7Ozs7Ozs7OztBQVdBOzs7Ozs7Ozs7QUFTQTs7Ozs7Ozs7O0FBU0E7Ozs7Ozs7O0FBUUE7Ozs7Ozs7O0FBUUFDLGdCQUFjQyxjQUFkLEVBQThCQyxLQUE5QixFQUFxQ0MsTUFBckMsRUFBNkM7QUFDM0MsUUFBSSxLQUFLaEIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsZUFBMUIsQ0FBUDtBQUNEO0FBQ0QsUUFBSWdCLEtBQUssRUFBVDtBQUNBLFFBQUdILG1CQUFtQkksU0FBdEIsRUFBaUM7QUFDL0JELFNBQUdILGNBQUgsR0FBb0JBLGNBQXBCO0FBQ0Q7QUFDRCxRQUFHQyxVQUFVRyxTQUFiLEVBQXdCO0FBQ3RCRCxTQUFHRixLQUFILEdBQVdBLEtBQVg7QUFDRDtBQUNELFFBQUdDLFdBQVdFLFNBQWQsRUFBeUI7QUFDdkJELFNBQUdELE1BQUgsR0FBWUEsTUFBWjtBQUNEO0FBQ0QsVUFBTWQsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSx5Q0FEUjtBQUVYTSxjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YVyxRQU5XO0FBT1hWLFlBQU07QUFQSyxLQUFiO0FBU0EsV0FBTyxLQUFLQyxXQUFMLENBQWlCQyxPQUFqQixDQUF5QlAsSUFBekIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7QUFNQWlCLGNBQVlDLFVBQVosRUFBd0I7QUFDdEIsUUFBSSxLQUFLcEIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsYUFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSwyQ0FBMENzQixVQUFXLEVBRDdEO0FBRVhoQixjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YQyxZQUFNO0FBTkssS0FBYjtBQVFBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7OztBQVVBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBb0RBOzs7Ozs7O0FBT0FtQixpQkFBZUQsVUFBZixFQUEyQkUsUUFBM0IsRUFBcUM7QUFDbkMsUUFBSSxLQUFLdEIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsZ0JBQTFCLENBQVA7QUFDRDtBQUNELFVBQU1DLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtMLEtBQU0sMkNBQTBDc0IsVUFBVyxFQUQ3RDtBQUVYaEIsY0FBUSxLQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWGlCLFlBQU1ELFFBTks7QUFPWGYsWUFBTTtBQVBLLEtBQWI7QUFTQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUCxJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBaUJBOzs7Ozs7O0FBT0FzQixpQkFBZUosVUFBZixFQUEyQkssaUJBQTNCLEVBQThDO0FBQzVDLFFBQUksS0FBS3pCLGNBQUwsRUFBSixFQUEyQjtBQUN6QixhQUFPLEtBQUtDLG9CQUFMLENBQTBCLGdCQUExQixDQUFQO0FBQ0Q7QUFDRCxVQUFNQyxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLTCxLQUFNLDJDQUEwQ3NCLFVBQVcsRUFEN0Q7QUFFWGhCLGNBQVEsUUFGRztBQUdYQyxlQUFTO0FBQ1Asc0JBQWMsS0FBS0M7QUFEWixPQUhFO0FBTVhpQixZQUFNRSxpQkFOSztBQU9YbEIsWUFBTTtBQVBLLEtBQWI7QUFTQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUCxJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQWdEQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQTJDQTs7Ozs7Ozs7Ozs7Ozs7Ozs7QUFpQkE7Ozs7Ozs7OztBQVNBd0IseUJBQXVCWixjQUF2QixFQUF1Q0MsS0FBdkMsRUFBOENDLE1BQTlDLEVBQXNEO0FBQ3BELFFBQUksS0FBS2hCLGNBQUwsRUFBSixFQUEyQjtBQUN6QixhQUFPLEtBQUtDLG9CQUFMLENBQTBCLHdCQUExQixDQUFQO0FBQ0Q7QUFDRCxRQUFJZ0IsS0FBSyxFQUFUO0FBQ0EsUUFBR0gsbUJBQW1CSSxTQUF0QixFQUFpQztBQUMvQkQsU0FBR0gsY0FBSCxHQUFvQkEsY0FBcEI7QUFDRDtBQUNELFFBQUdDLFVBQVVHLFNBQWIsRUFBd0I7QUFDdEJELFNBQUdGLEtBQUgsR0FBV0EsS0FBWDtBQUNEO0FBQ0QsUUFBR0MsV0FBV0UsU0FBZCxFQUF5QjtBQUN2QkQsU0FBR0QsTUFBSCxHQUFZQSxNQUFaO0FBQ0Q7QUFDRCxVQUFNZCxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLTCxLQUFNLG1EQURSO0FBRVhNLGNBQVEsS0FGRztBQUdYQyxlQUFTO0FBQ1Asc0JBQWMsS0FBS0M7QUFEWixPQUhFO0FBTVhXLFFBTlc7QUFPWFYsWUFBTTtBQVBLLEtBQWI7QUFTQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUCxJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7OztBQU1BeUIsdUJBQXFCQyxXQUFyQixFQUFrQztBQUNoQyxRQUFJLEtBQUs1QixjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixzQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSxxREFBb0Q4QixXQUFZLEVBRHhFO0FBRVh4QixjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YQyxZQUFNO0FBTkssS0FBYjtBQVFBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7OztBQU9BMkIsMEJBQXdCRCxXQUF4QixFQUFxQ0UsU0FBckMsRUFBZ0Q7QUFDOUMsUUFBSSxLQUFLOUIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIseUJBQTFCLENBQVA7QUFDRDtBQUNELFVBQU1DLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtMLEtBQU0scURBQW9EOEIsV0FBWSxFQUR4RTtBQUVYeEIsY0FBUSxLQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWGlCLFlBQU1PLFNBTks7QUFPWHZCLFlBQU07QUFQSyxLQUFiO0FBU0EsV0FBTyxLQUFLQyxXQUFMLENBQWlCQyxPQUFqQixDQUF5QlAsSUFBekIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7O0FBT0E2QiwwQkFBd0JILFdBQXhCLEVBQXFDSCxpQkFBckMsRUFBd0Q7QUFDdEQsUUFBSSxLQUFLekIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIseUJBQTFCLENBQVA7QUFDRDtBQUNELFVBQU1DLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtMLEtBQU0scURBQW9EOEIsV0FBWSxFQUR4RTtBQUVYeEIsY0FBUSxRQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWGlCLFlBQU1FLGlCQU5LO0FBT1hsQixZQUFNO0FBUEssS0FBYjtBQVNBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7QUFRQThCLGdDQUE4QkosV0FBOUIsRUFBMkNSLFVBQTNDLEVBQXVESyxpQkFBdkQsRUFBMEU7QUFDeEUsUUFBSSxLQUFLekIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsK0JBQTFCLENBQVA7QUFDRDtBQUNELFVBQU1DLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtMLEtBQU0scURBQW9EOEIsV0FBWSxZQUFXUixVQUFXLEVBRDlGO0FBRVhoQixjQUFRLFFBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YaUIsWUFBTUUsaUJBTks7QUFPWGxCLFlBQU07QUFQSyxLQUFiO0FBU0EsV0FBTyxLQUFLQyxXQUFMLENBQWlCQyxPQUFqQixDQUF5QlAsSUFBekIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7OztBQVFBK0IsaUJBQWVuQixjQUFmLEVBQStCQyxLQUEvQixFQUFzQ0MsTUFBdEMsRUFBOEM7QUFDNUMsUUFBSSxLQUFLaEIsY0FBTCxFQUFKLEVBQTJCO0FBQ3pCLGFBQU8sS0FBS0Msb0JBQUwsQ0FBMEIsZ0JBQTFCLENBQVA7QUFDRDtBQUNELFFBQUlnQixLQUFLLEVBQVQ7QUFDQSxRQUFHSCxtQkFBbUJJLFNBQXRCLEVBQWlDO0FBQy9CRCxTQUFHSCxjQUFILEdBQW9CQSxjQUFwQjtBQUNEO0FBQ0QsUUFBR0MsVUFBVUcsU0FBYixFQUF3QjtBQUN0QkQsU0FBR0YsS0FBSCxHQUFXQSxLQUFYO0FBQ0Q7QUFDRCxRQUFHQyxXQUFXRSxTQUFkLEVBQXlCO0FBQ3ZCRCxTQUFHRCxNQUFILEdBQVlBLE1BQVo7QUFDRDtBQUNELFVBQU1kLE9BQU87QUFDWEMsV0FBTSxHQUFFLEtBQUtMLEtBQU0sMENBRFI7QUFFWE0sY0FBUSxLQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWFcsUUFOVztBQU9YVixZQUFNO0FBUEssS0FBYjtBQVNBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7O0FBTUFnQyxnQkFBY0MsWUFBZCxFQUE0QjtBQUMxQixRQUFJLEtBQUtuQyxjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixlQUExQixDQUFQO0FBQ0Q7QUFDRCxVQUFNQyxPQUFPO0FBQ1hDLFdBQU0sR0FBRSxLQUFLTCxLQUFNLDRDQUEyQ3FDLFlBQWEsRUFEaEU7QUFFWC9CLGNBQVEsS0FGRztBQUdYQyxlQUFTO0FBQ1Asc0JBQWMsS0FBS0M7QUFEWixPQUhFO0FBTVhDLFlBQU07QUFOSyxLQUFiO0FBUUEsV0FBTyxLQUFLQyxXQUFMLENBQWlCQyxPQUFqQixDQUF5QlAsSUFBekIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7O0FBT0FrQyxtQkFBaUJELFlBQWpCLEVBQStCRSxVQUEvQixFQUEyQztBQUN6QyxRQUFJLEtBQUtyQyxjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixrQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSw0Q0FBMkNxQyxZQUFhLEVBRGhFO0FBRVgvQixjQUFRLEtBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YaUIsWUFBTWMsVUFOSztBQU9YOUIsWUFBTTtBQVBLLEtBQWI7QUFTQSxXQUFPLEtBQUtDLFdBQUwsQ0FBaUJDLE9BQWpCLENBQXlCUCxJQUF6QixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7QUFPQW9DLG1CQUFpQkgsWUFBakIsRUFBK0JWLGlCQUEvQixFQUFrRDtBQUNoRCxRQUFJLEtBQUt6QixjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixrQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSw0Q0FBMkNxQyxZQUFhLEVBRGhFO0FBRVgvQixjQUFRLFFBRkc7QUFHWEMsZUFBUztBQUNQLHNCQUFjLEtBQUtDO0FBRFosT0FIRTtBQU1YaUIsWUFBTUUsaUJBTks7QUFPWGxCLFlBQU07QUFQSyxLQUFiO0FBU0EsV0FBTyxLQUFLQyxXQUFMLENBQWlCQyxPQUFqQixDQUF5QlAsSUFBekIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7OztBQVFBcUMscUJBQW1CSixZQUFuQixFQUFpQ2YsVUFBakMsRUFBNkNLLGlCQUE3QyxFQUFnRTtBQUM5RCxRQUFJLEtBQUt6QixjQUFMLEVBQUosRUFBMkI7QUFDekIsYUFBTyxLQUFLQyxvQkFBTCxDQUEwQixvQkFBMUIsQ0FBUDtBQUNEO0FBQ0QsVUFBTUMsT0FBTztBQUNYQyxXQUFNLEdBQUUsS0FBS0wsS0FBTSw0Q0FBMkNxQyxZQUFhLGtCQUFpQmYsVUFBVyxFQUQ1RjtBQUVYaEIsY0FBUSxRQUZHO0FBR1hDLGVBQVM7QUFDUCxzQkFBYyxLQUFLQztBQURaLE9BSEU7QUFNWGlCLFlBQU1FLGlCQU5LO0FBT1hsQixZQUFNO0FBUEssS0FBYjtBQVNBLFdBQU8sS0FBS0MsV0FBTCxDQUFpQkMsT0FBakIsQ0FBeUJQLElBQXpCLENBQVA7QUFDRDs7QUE1eUI0RDtrQkFBMUNWLG1CIiwiZmlsZSI6ImNvbmZpZ3VyYXRpb24uY2xpZW50LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5pbXBvcnQgTWV0YUFwaUNsaWVudCBmcm9tICcuLi9tZXRhQXBpLmNsaWVudCc7XG5pbXBvcnQgVGltZW91dEVycm9yIGZyb20gJy4uL3RpbWVvdXRFcnJvcic7XG5pbXBvcnQgcmFuZG9tc3RyaW5nIGZyb20gJ3JhbmRvbXN0cmluZyc7XG5cbi8qKlxuICogbWV0YWFwaS5jbG91ZCBDb3B5RmFjdG9yeSBjb25maWd1cmF0aW9uIEFQSSAodHJhZGUgY29weWluZyBjb25maWd1cmF0aW9uIEFQSSkgY2xpZW50IChzZWVcbiAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5LylcbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgQ29uZmlndXJhdGlvbkNsaWVudCBleHRlbmRzIE1ldGFBcGlDbGllbnQge1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIENvcHlGYWN0b3J5IGNvbmZpZ3VyYXRpb24gQVBJIGNsaWVudCBpbnN0YW5jZVxuICAgKiBAcGFyYW0ge0h0dHBDbGllbnR9IGh0dHBDbGllbnQgSFRUUCBjbGllbnRcbiAgICogQHBhcmFtIHtTdHJpbmd9IHRva2VuIGF1dGhvcml6YXRpb24gdG9rZW5cbiAgICogQHBhcmFtIHtTdHJpbmd9IGRvbWFpbiBkb21haW4gdG8gY29ubmVjdCB0bywgZGVmYXVsdCBpcyBhZ2lsaXVtdHJhZGUuYWdpbGl1bXRyYWRlLmFpXG4gICAqL1xuICBjb25zdHJ1Y3RvcihodHRwQ2xpZW50LCB0b2tlbiwgZG9tYWluID0gJ2FnaWxpdW10cmFkZS5hZ2lsaXVtdHJhZGUuYWknKSB7XG4gICAgc3VwZXIoaHR0cENsaWVudCwgdG9rZW4sIGRvbWFpbik7XG4gICAgdGhpcy5faG9zdCA9IGBodHRwczovL2NvcHlmYWN0b3J5LWFwcGxpY2F0aW9uLWhpc3RvcnktbWFzdGVyLXYxLiR7ZG9tYWlufWA7XG4gIH1cblxuICAvKipcbiAgICogU3RyYXRlZ3kgaWRcbiAgICogQHR5cGVkZWYge09iamVjdH0gU3RyYXRlZ3lJZFxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gaWQgc3RyYXRlZ3kgaWRcbiAgICovXG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBuZXcgdW51c2VkIHN0cmF0ZWd5IGlkLiBNZXRob2QgaXMgYWNjZXNzaWJsZSBvbmx5IHdpdGggQVBJIGFjY2VzcyB0b2tlbi4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vZ2VuZXJhdGVTdHJhdGVneUlkL1xuICAgKiBAcmV0dXJuIHtQcm9taXNlPFN0cmF0ZWd5SWQ+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIHN0cmF0ZWd5IGlkIGdlbmVyYXRlZFxuICAgKi9cbiAgZ2VuZXJhdGVTdHJhdGVneUlkKCkge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdnZW5lcmF0ZVN0cmF0ZWd5SWQnKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIHVybDogYCR7dGhpcy5faG9zdH0vdXNlcnMvY3VycmVudC9jb25maWd1cmF0aW9uL3VudXNlZC1zdHJhdGVneS1pZGAsXG4gICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgcmFuZG9tIGFjY291bnQgaWRcbiAgICogQHJldHVybiB7U3RyaW5nfSBhY2NvdW50IGlkXG4gICAqL1xuICBnZW5lcmF0ZUFjY291bnRJZCgpIHtcbiAgICByZXR1cm4gcmFuZG9tc3RyaW5nLmdlbmVyYXRlKDY0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSBzdHJhdGVneSBzdWJzY3JpcHRpb25zXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lTdWJzY3JpcHRpb25cbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IHN0cmF0ZWd5SWQgaWQgb2YgdGhlIHN0cmF0ZWd5IHRvIHN1YnNjcmliZSB0b1xuICAgKiBAcHJvcGVydHkge051bWJlcn0gW211bHRpcGxpZXJdIHN1YnNjcmlwdGlvbiBtdWx0aXBsaWVyLCBkZWZhdWx0IGlzIDF4XG4gICAqIEBwcm9wZXJ0eSB7Qm9vbGVhbn0gW3NraXBQZW5kaW5nT3JkZXJzXSBmbGFnIGluZGljYXRpbmcgdGhhdCBwZW5kaW5nIG9yZGVycyBzaG91bGQgbm90IGJlIGNvcGllZC4gRGVmYXVsdFxuICAgKiBpcyB0byBjb3B5IHBlbmRpbmcgb3JkZXJzXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBbY2xvc2VPbmx5XSBzZXR0aW5nIHdpY2ggaW5zdHJ1Y3RzIHRoZSBhcHBsaWNhdGlvbiBub3QgdG8gb3BlbiBuZXcgcG9zaXRpb25zLiBieS1zeW1ib2xcbiAgICogbWVhbnMgdGhhdCBpdCBpcyBzdGlsbCBhbGxvd2VkIHRvIG9wZW4gbmV3IHBvc2l0aW9ucyB3aXRoIGEgc3ltYm9sIGVxdWFsIHRvIHRoZSBzeW1ib2wgb2YgYW4gZXhpc3Rpbmcgc3RyYXRlZ3lcbiAgICogcG9zaXRpb24gKGNhbiBiZSB1c2VkIHRvIGdyYWNlZnVseSBleGl0IHN0cmF0ZWdpZXMgdHJhZGluZyBpbiBuZXR0aW5nIG1vZGUgb3IgcGxhY2luZyBhIHNlcmllcyBvZiByZWxhdGVkIHRyYWRlc1xuICAgKiBwZXIgc3ltYm9sKS4gaW1tZWRpYXRlbHkgbWVhbnMgdG8gY2xvc2UgYWxsIHBvc2l0aW9ucyBpbW1lZGlhdGVseS4gT25lIG9mICdieS1wb3NpdGlvbicsICdieS1zeW1ib2wnLCAnaW1tZWRpYXRlbHknXG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbbWF4VHJhZGVSaXNrXSBtYXggcmlzayBwZXIgdHJhZGUsIGV4cHJlc3NlZCBhcyBhIGZyYWN0aW9uIG9mIDEuIElmIHRyYWRlIGhhcyBhIFNMLCB0aGVcbiAgICogdHJhZGUgc2l6ZSB3aWxsIGJlIGFkanVzdGVkIHRvIG1hdGNoIHRoZSByaXNrIGxpbWl0LiBJZiBub3QsIHRoZSB0cmFkZSBTTCB3aWxsIGJlIGFwcGxpZWQgYWNjb3JkaW5nIHRvIHRoZSByaXNrXG4gICAqIGxpbWl0XG4gICAqIEBwcm9wZXJ0eSB7Qm9vbGVhbn0gW3JldmVyc2VdIGZsYWcgaW5kaWNhdGluZyB0aGF0IHRoZSBzdHJhdGVneSBzaG91bGQgYmUgY29waWVkIGluIGEgcmV2ZXJzZSBkaXJlY3Rpb25cbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IFtyZWR1Y2VDb3JyZWxhdGlvbnNdIHNldHRpbmcgaW5kaWNhdGluZyB3aGV0aGVyIHRvIGVuYWJsZSBhdXRvbWF0aWMgdHJhZGVcbiAgICogY29ycmVsYXRpb24gcmVkdWN0aW9uLiBQb3NzaWJsZSBzZXR0aW5ncyBhcmUgbm90IHNwZWNpZmllZCAoZGlzYWJsZSBjb3JyZWxhdGlvbiByaXNrIHJlc3RyaWN0aW9ucyksXG4gICAqIGJ5LXN0cmF0ZWd5IChsaW1pdCBjb3JyZWxhdGlvbnMgZm9yIHRoZSBzdHJhdGVneSkgb3IgYnktYWNjb3VudCAobGltaXQgY29ycmVsYXRpb25zIGZvciB0aGUgYWNjb3VudClcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5U3RvcE91dFNldHRpbmdzfSBbc3RvcE91dFJpc2tdIHN0b3Agb3V0IHNldHRpbmcuIEFsbCB0cmFkaW5nIHdpbGwgYmUgdGVybWluYXRlZFxuICAgKiBhbmQgcG9zaXRpb25zIGNsb3NlZCBvbmNlIGVxdWl0eSBkcmF3ZG93biByZWFjaGVzIHRoaXMgdmFsdWVcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5U3ltYm9sRmlsdGVyfSBbc3ltYm9sRmlsdGVyXSBzeW1ib2wgZmlsdGVyIHdoaWNoIGNhbiBiZSB1c2VkIHRvIGNvcHkgb25seSBzcGVjaWZpY1xuICAgKiBzeW1ib2xzIG9yIGV4Y2x1ZGUgc29tZSBzeW1ib2xzIGZyb20gY29weWluZ1xuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lOZXdzRmlsdGVyfSBbbmV3c0ZpbHRlcl0gbmV3cyByaXNrIGZpbHRlciBjb25maWd1cmF0aW9uXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8Q29weUZhY3RvcnlTdHJhdGVneVJpc2tMaW1pdD59IFtyaXNrTGltaXRzXSBzdHJhdGVneSByaXNrIGxpbWl0cy4gWW91IGNhbiBjb25maWd1cmUgdHJhZGluZyB0byBiZVxuICAgKiBzdG9wcGVkIG9uY2UgdG90YWwgZHJhd2Rvd24gZ2VuZXJhdGVkIGR1cmluZyBzcGVjaWZpYyBwZXJpb2QgaXMgZXhjZWVkZWQuIENhbiBiZSBzcGVjaWZpZWQgZWl0aGVyIGZvciBiYWxhbmNlIG9yXG4gICAqIGVxdWl0eSBkcmF3ZG93blxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lNYXhTdG9wTG9zc30gW21heFN0b3BMb3NzXSBzdG9wIGxvc3MgdmFsdWUgcmVzdHJpY3Rpb25cbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhMZXZlcmFnZV0gc2V0dGluZyBpbmRpY2F0aW5nIG1heGltdW0gbGV2ZXJhZ2UgYWxsb3dlZCB3aGVuIG9wZW5pbmcgYSBuZXcgcG9zaXRpb25zLlxuICAgKiBBbnkgdHJhZGUgd2hpY2ggcmVzdWx0cyBpbiBhIGhpZ2hlciBsZXZlcmFnZSB3aWxsIGJlIGRpc2NhcmRlZFxuICAgKiBAcHJvcGVydHkge0FycmF5PENvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xNYXBwaW5nPn0gW3N5bWJvbE1hcHBpbmddIGRlZmluZXMgaG93IHN5bWJvbCBuYW1lIHNob3VsZCBiZSBjaGFuZ2VkIHdoZW5cbiAgICogdHJhZGluZyAoZS5nLiB3aGVuIGJyb2tlciB1c2VzIHN5bWJvbCBuYW1lcyB3aXRoIHVudXN1YWwgc3VmZml4ZXMpLiBCeSBkZWZhdWx0IHRoaXMgc2V0dGluZyBpcyBkaXNhYmxlZCBhbmQgdGhlXG4gICAqIHRyYWRlcyBhcmUgY29waWVkIHVzaW5nIHNpZ25hbCBzb3VyY2Ugc3ltYm9sIG5hbWVcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5VHJhZGVTaXplU2NhbGluZ30gW3RyYWRlU2l6ZVNjYWxpbmddIFRyYWRlIHNpemUgc2NhbGluZyBzZXR0aW5ncy4gQnkgZGVmYXVsdCB0aGVcbiAgICogdHJhZGUgc2l6ZSBvbiBzdHJhdGVneSBzdWJzY3JpYmVyIHNpZGUgd2lsbCBiZSBzY2FsZWQgYWNjb3JkaW5nIHRvIGJhbGFuY2UgdG8gcHJlc2VydmUgcmlzay5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVN0b3BMb3NzXSBmbGFnIGluZGljYXRpbmcgd2hldGhlciBzdG9wIGxvc3Mgc2hvdWxkIGJlIGNvcGllZC4gRGVmYXVsdCBpcyB0byBjb3B5IHN0b3BcbiAgICogbG9zcy5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVRha2VQcm9maXRdIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIHRha2UgcHJvZml0IHNob3VsZCBiZSBjb3BpZWQuIERlZmF1bHQgaXMgdG8gY29weSB0YWtlXG4gICAqIHByb2ZpdC5cbiAgICogQHByb3BlcnR5IHtudW1iZXJ9IFttaW5UcmFkZVZvbHVtZV0gTWluaW11bSB0cmFkZSB2b2x1bWUgdG8gY29weS4gVHJhZGUgc2lnbmFscyB3aXRoIGEgc21hbGxlciB2b2x1bWUgd2lsbCBub3QgYmVcbiAgICogY29waWVkXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbbWF4VHJhZGVWb2x1bWVdIE1heGltdW0gdHJhZGUgdm9sdW1lIHRvIGNvcHkuIFRyYWRlIHNpZ25hbHMgd2l0aCBhIGxhcmdlciB2b2x1bWUgd2lsbCBiZSBjb3BpZWRcbiAgICogd2l0aCBtYXhpbXVtIHZvbHVtZSBpbnN0ZWFkXG4gICAqIEBwcm9wZXJ0eSB7Ym9vbGVhbn0gW3JlbW92ZWRdIGZsYWcgaW5kaWNhdGluZyB0aGF0IHRoZSBzdWJzY3JpcHRpb24gd2FzIHNjaGVkdWxlZCBmb3IgcmVtb3ZhbCBvbmNlIGFsbCBzdWJzY3JpcHRpb25cbiAgICogcG9zaXRpb25zIHdpbGwgYmUgY2xvc2VkXG4gICAqL1xuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSBzdHJhdGVneSB0cmFkZSBzaXplIHNjYWxpbmcgc2V0dGluZ3NcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlTdHJhdGVneVRyYWRlU2l6ZVNjYWxpbmdcbiAgICogQHByb3BlcnR5IHtzdHJpbmd9IG1vZGUgSWYgc2V0IHRvIGJhbGFuY2UsIHRoZSB0cmFkZSBzaXplIG9uIHN0cmF0ZWd5IHN1YnNjcmliZXIgd2lsbCBiZSBzY2FsZWQgYWNjb3JkaW5nIHRvXG4gICAqIGJhbGFuY2UgdG8gcHJlc2VydmUgcmlzay4gSWYgdmFsdWUgaXMgbm9uZSwgdGhlbiB0cmFkZSBzaXplIHdpbGwgYmUgcHJlc2VydmVkIGlycmVnYXJkbGVzcyBvZiB0aGUgc3Vic2NyaWJlclxuICAgKiBiYWxhbmNlLiBJZiB2YWx1ZSBpcyBjb250cmFjdFNpemUsIHRoZW4gdHJhZGUgc2l6ZSB3aWxsIGJlIHNjYWxlZCBhY2NvcmRpbmcgdG8gY29udHJhY3Qgc2l6ZS4gSWYgZml4ZWRWb2x1bWUgaXNcbiAgICogc2V0LCB0aGVuIHRyYWRlIHdpbGwgYmUgY29waWVkIHdpdGggYSBmaXhlZCB2b2x1bWUgb2YgdHJhY2VWb2x1bWUgc2V0dGluZy4gSWYgZml4ZWRSaXNrIGlzIHNldCwgdGhlbiBlYWNoIHRyYWRlXG4gICAqIHdpbGwgYmUgY29waWVkIHdpdGggYSB0cmFkZSB2b2x1bWUgc2V0IHRvIHJpc2sgc3BlY2lmaWMgZnJhY3Rpb24gb2YgYmFsYW5jZSBhcyBjb25maWd1cmVkIGJ5IHJpc2tGcmFjdGlvbiBzZXR0aW5nLlxuICAgKiBOb3RlLCB0aGF0IGluIGZpeGVkUmlzayBtb2RlIHRyYWRlcyB3aXRob3V0IGEgU0wgYXJlIG5vdCBjb3BpZWQuIERlZmF1bHQgaXMgYmFsYW5jZS4gQWxsb3dlZCB2YWx1ZXM6IG5vbmUsXG4gICAqIGNvbnRyYWN0U2l6ZSwgYmFsYW5jZSwgZml4ZWRWb2x1bWUsIGZpeGVkUmlza1xuICAgKiBAcHJvcGVydHkge251bWJlcn0gW3RyYWRlVm9sdW1lXSBGaXhlZCB0cmFkZSB2b2x1bWUgZm9yIHVzZSB3aXRoIGZpeGVkVm9sdW1lIHRyYWRlIHNpemUgc2NhbGluZyBtb2RlXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbcmlza0ZyYWN0aW9uXSBGaXhlZCByaXNrIGZyYWN0aW9uIGZvciB1c2Ugd2l0aCBmaXhlZFJpc2sgdHJhZGUgc2l6ZSBzY2FsaW5nIG1vZGVcbiAgICovXG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IHN0cmF0ZWd5IHN0b3BvdXQgc2V0dGluZ3NcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlTdHJhdGVneVN0b3BPdXRTZXR0aW5nc1xuICAgKiBAcHJvcGVydHkge051bWJlcn0gW2Fic29sdXRlVmFsdWVdIHZhbHVlIG9mIHRoZSBzdG9wIG91dCByaXNrLCBtZWFzdXJlZCBpbiBhY2NvdW50IGN1cnJlbmN5XG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbcmVsYXRpdmVWYWx1ZV0gdmFsdWUgb2YgdGhlIHN0b3Agb3V0IHJpc2ssIGV4cHJlc3NlZCBhcyBhIGZyYWN0aW9uIG9mIDFcbiAgICogQHByb3BlcnR5IHtEYXRlfHN0cmluZ3xtb21lbnQuTW9tZW50fSBbc3RhcnRUaW1lXSB0aGUgdGltZSB0byBzdGFydCByaXNrIGNhbGN1bGF0aW9uIGZyb20uIEFsbCBwcmV2aW91cyB0cmFkZXMgd2lsbCBiZSBpZ25vcmVkLiBZb3UgY2FuXG4gICAqIHVzZSBpdCB0byByZXNldCB0aGUgcmlzayBjb3VudGVyIGFmdGVyIGEgc3RvcG91dCBldmVudFxuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3Rvcnkgc3ltYm9sIGZpbHRlclxuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5U3ltYm9sRmlsdGVyXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8U3RyaW5nPn0gaW5jbHVkZWQgbGlzdCBvZiBzeW1ib2xzIGNvcGllZC4gTGVhdmUgdGhlIHZhbHVlIGVtcHR5IHRvIGNvcHkgYWxsIHN5bWJvbHNcbiAgICogQHByb3BlcnR5IHtBcnJheTxTdHJpbmc+fSBleGNsdWRlZCBsaXN0IG9mIHN5bWJvbHMgZXhjbHVkZWQgZnJvbSBjb3B5aW5nLiBMZWF2ZSB0aGUgdmFsdWUgZW1wdHkgdG8gY29weSBhbGwgc3ltYm9sc1xuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3RvcnkgbmV3cyByaXNrIGZpbHRlclxuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5TmV3c0ZpbHRlclxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lCcmVha2luZ05ld3NGaWx0ZXJ9IFticmVha2luZ05ld3NGaWx0ZXJdIGJyZWFraW5nIG5ld3MgZmlsdGVyXG4gICAqIEBwcm9wZXJ0eSB7Q29weUZhY3RvcnlTdHJhdGVneUNhbGVuZGFyTmV3c0ZpbHRlcn0gW2NhbGVuZGFyTmV3c0ZpbHRlcl0gY2FsZW5kYXIgbmV3cyBmaWx0ZXJcbiAgICovXG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IGJyZWFraW5nIG5ld3MgcmlzayBmaWx0ZXJcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlTdHJhdGVneUJyZWFraW5nTmV3c0ZpbHRlclxuICAgKiBAcHJvcGVydHkge0FycmF5PFN0cmluZz59IHByaW9yaXRpZXMgbGlzdCBvZiBicmVha2luZyBuZXdzIHByaW9yaXRpZXMgdG8gc3RvcCB0cmFkaW5nIG9uLCBsZWF2ZSBlbXB0eSB0byBkaXNhYmxlXG4gICAqIGJyZWFraW5nIG5ld3MgZmlsdGVyLiBPbmUgb2YgaGlnaCwgbWVkaXVtLCBsb3cuXG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbY2xvc2VQb3NpdGlvblRpbWVHYXBJbk1pbnV0ZXNdIHRpbWUgaW50ZXJ2YWwgc3BlY2lmeWluZyB3aGVuIHRvIGZvcmNlIGNsb3NlIGFuIGFscmVhZHlcbiAgICogb3BlbiBwb3NpdGlvbiBhZnRlciBicmVha2luZyBuZXdzLiBEZWZhdWx0IHZhbHVlIGlzIDYwIG1pbnV0ZXNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFtvcGVuUG9zaXRpb25Gb2xsb3dpbmdUaW1lR2FwSW5NaW51dGVzXSB0aW1lIGludGVydmFsIHNwZWNpZnlpbmcgd2hlbiBpdCBpcyBhbGxvd2VkIHRvXG4gICAqIG9wZW4gcG9zaXRpb24gYWZ0ZXIgY2FsZW5kYXIgbmV3cy4gRGVmYXVsdCB2YWx1ZSBpcyA2MCBtaW51dGVzXG4gICAqL1xuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSBjYWxlbmRhciBuZXcgZmlsdGVyXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lDYWxlbmRhck5ld3NGaWx0ZXJcbiAgICogQHByb3BlcnR5IHtBcnJheTxTdHJpbmc+fSBwcmlvcml0aWVzIGxpc3Qgb2YgY2FsZW5kYXIgbmV3cyBwcmlvcml0aWVzIHRvIHN0b3AgdHJhZGluZyBvbiwgbGVhdmUgZW1wdHkgdG8gZGlzYWJsZVxuICAgKiBjYWxlbmRhciBuZXdzIGZpbHRlci4gT25lIG9mIGVsZWN0aW9uLCBoaWdoLCBtZWRpdW0sIGxvdy5cbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFtjbG9zZVBvc2l0aW9uVGltZUdhcEluTWludXRlc10gdGltZSBpbnRlcnZhbCBzcGVjaWZ5aW5nIHdoZW4gdG8gZm9yY2UgY2xvc2UgYW4gYWxyZWFkeVxuICAgKiBvcGVuIHBvc2l0aW9uIGJlZm9yZSBjYWxlbmRhciBuZXdzLiBEZWZhdWx0IHZhbHVlIGlzIDYwIG1pbnV0ZXNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFtvcGVuUG9zaXRpb25QcmVjZWRpbmdUaW1lR2FwSW5NaW51dGVzXSB0aW1lIGludGVydmFsIHNwZWNpZnlpbmcgd2hlbiBpdCBpcyBzdGlsbFxuICAgKiBhbGxvd2VkIHRvIG9wZW4gcG9zaXRpb24gYmVmb3JlIGNhbGVuZGFyIG5ld3MuIERlZmF1bHQgdmFsdWUgaXMgMTIwIG1pbnV0ZXNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFtvcGVuUG9zaXRpb25Gb2xsb3dpbmdUaW1lR2FwSW5NaW51dGVzXSB0aW1lIGludGVydmFsIHNwZWNpZnlpbmcgd2hlbiBpdCBpcyBhbGxvd2VkIHRvXG4gICAqIG9wZW4gcG9zaXRpb24gYWZ0ZXIgY2FsZW5kYXIgbmV3cy4gRGVmYXVsdCB2YWx1ZSBpcyA2MCBtaW51dGVzXG4gICAqL1xuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSByaXNrIGxpbWl0IGZpbHRlclxuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5Umlza0xpbWl0XG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSB0eXBlIHJlc3RyaWN0aW9uIHR5cGUuIE9uZSBvZiBkYWlseSwgbW9udGhseSwgb3IgeWVhcmx5XG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBhcHBseVRvIGFjY291bnQgbWV0cmljIHRvIGFwcGx5IGxpbWl0IHRvLiBPbmUgb2YgYmFsYW5jZSwgZXF1aXR5XG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbbWF4QWJzb2x1dGVSaXNrXSBtYXggZHJhd2Rvd24gYWxsb3dlZCwgbWVhc3VyZWQgaW4gYWNjb3VudCBjdXJyZW5jeVxuICAgKiBAcHJvcGVydHkge051bWJlcn0gW21heFJlbGF0aXZlUmlza10gbWF4IGRyYXdkb3duIGFsbG93ZWQsIG1lYXN1cmVkIGluIGFjY291bnQgY3VycmVuY3lcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBjbG9zZVBvc2l0aW9ucyB3aGV0aGVyIHRvIGZvcmNlIGNsb3NlIHBvc2l0aW9ucyB3aGVuIHRoZSByaXNrIGlzIHJlYWNoZWQuIElmIHZhbHVlIGlzIGZhbHNlXG4gICAqIHRoZW4gb25seSB0aGUgbmV3IHRyYWRlcyB3aWxsIGJlIGhhbHRlZCwgYnV0IGV4aXN0aW5nIG9uZXMgd2lsbCBub3QgYmUgY2xvc2VkXG4gICAqIEBwcm9wZXJ0eSB7RGF0ZXxzdHJpbmd8bW9tZW50Lk1vbWVudH0gW3N0YXJ0VGltZV0gdGltZSB0byBzdGFydCByaXNrIHRyYWNraW5nIGZyb20uIEFsbCBwcmV2aW91cyB0cmFkZXMgd2lsbCBiZSBpZ25vcmVkLiBZb3VcbiAgICogY2FuIHVzZSB0aGlzIHZhbHVlIHRvIHJlc2V0IHRoZSBmaWx0ZXIgYWZ0ZXIgc3RvcG91dCBldmVudFxuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3Rvcnkgc3RyYXRlZ3kgbWF4IHN0b3AgbG9zcyBzZXR0aW5nc1xuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5TWF4U3RvcExvc3NcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IHZhbHVlIG1heGltdW0gU0wgdmFsdWVcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IHVuaXRzIFNMIHVuaXRzLiBPbmx5IHBpcHMgdmFsdWUgaXMgc3VwcG9ydGVkIGF0IHRoaXMgcG9pbnRcbiAgICovXG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IHN0cmF0ZWd5IHN5bWJvbCBtYXBwaW5nXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xNYXBwaW5nXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBmcm9tIHN5bWJvbCBuYW1lIHRvIGNvbnZlcnQgZnJvbVxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gdG8gc3ltYm9sIG5hbWUgdG8gY29udmVydCB0b1xuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3RvcnkgYWNjb3VudCB1cGRhdGVcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlTdWJzY3JpYmVyVXBkYXRlXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBuYW1lIGFjY291bnQgaHVtYW4tcmVhZGFibGUgbmFtZVxuICAgKiBAcHJvcGVydHkge051bWJlcn0gW3Jlc2VydmVkTWFyZ2luRnJhY3Rpb25dIGZyYWN0aW9uIG9mIHJlc2VydmVkIG1hcmdpbiB0byByZWR1Y2UgYSByaXNrIG9mIG1hcmdpbiBjYWxsLlxuICAgKiBEZWZhdWx0IGlzIHRvIHJlc2VydmUgbm8gbWFyZ2luLiBXZSByZWNvbW1lbmQgdXNpbmcgbWF4TGV2ZXJhZ2Ugc2V0dGluZyBpbnN0ZWFkLiBTcGVjaWZpZWQgYXMgYSBmcmFjdGlvbiBvZiBiYWxhbmNlXG4gICAqIHRodXMgdGhlIHZhbHVlIGlzIHVzdWFsbHkgZ3JlYXRlciB0aGFuIDFcbiAgICogQHByb3BlcnR5IHtBcnJheTxTdHJpbmc+fSBbcGhvbmVOdW1iZXJzXSBwaG9uZSBudW1iZXJzIHRvIHNlbmQgc21zIG5vdGlmaWNhdGlvbnMgdG8uIExlYXZlIGVtcHR5IHRvXG4gICAqIHJlY2VpdmUgbm8gc21zIG5vdGlmaWNhdGlvbnNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttaW5UcmFkZUFtb3VudF0gdmFsdWUgb2YgbWluaW1hbCB0cmFkZSBzaXplIGFsbG93ZWQsIGV4cHJlc3NlZCBpbiBhbW91bnQgb2YgYWNjb3VudFxuICAgKiBjdXJyZW5jeS4gQ2FuIGJlIHVzZWZ1bCBpZiB5b3VyIGJyb2tlciBjaGFyZ2VzIGEgZml4ZWQgZmVlIHBlciB0cmFuc2FjdGlvbiBzbyB0aGF0IHlvdSBjYW4gc2tpcCBzbWFsbCB0cmFkZXMgd2l0aFxuICAgKiBoaWdoIGJyb2tlciBjb21taXNzaW9uIHJhdGVzLiBEZWZhdWx0IGlzIDBcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IFtjbG9zZU9ubHldIHNldHRpbmcgd2ljaCBpbnN0cnVjdHMgdGhlIGFwcGxpY2F0aW9uIG5vdCB0byBvcGVuIG5ldyBwb3NpdGlvbnMuIGJ5LXN5bWJvbFxuICAgKiBtZWFucyB0aGF0IGl0IGlzIHN0aWxsIGFsbG93ZWQgdG8gb3BlbiBuZXcgcG9zaXRpb25zIHdpdGggYSBzeW1ib2wgZXF1YWwgdG8gdGhlIHN5bWJvbCBvZiBhbiBleGlzdGluZyBzdHJhdGVneVxuICAgKiBwb3NpdGlvbiAoY2FuIGJlIHVzZWQgdG8gZ3JhY2VmdWx5IGV4aXQgc3RyYXRlZ2llcyB0cmFkaW5nIGluIG5ldHRpbmcgbW9kZSBvciBwbGFjaW5nIGEgc2VyaWVzIG9mIHJlbGF0ZWQgdHJhZGVzXG4gICAqIHBlciBzeW1ib2wpLiBpbW1lZGlhdGVseSBtZWFucyB0byBjbG9zZSBhbGwgcG9zaXRpb25zIGltbWVkaWF0ZWx5LiBPbmUgb2YgJ2J5LXBvc2l0aW9uJywgJ2J5LXN5bWJvbCcsICdpbW1lZGlhdGVseSdcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5U3RvcE91dFNldHRpbmdzfSBbc3RvcE91dFJpc2tdIHN0b3Agb3V0IHNldHRpbmcuIEFsbCB0cmFkaW5nIHdpbGxcbiAgICogYmUgdGVybWluYXRlZCBhbmQgcG9zaXRpb25zIGNsb3NlZCBvbmNlIGVxdWl0eSBkcmF3ZG93biByZWFjaGVzIHRoaXMgdmFsdWVcbiAgICogQHByb3BlcnR5IHtBcnJheTxDb3B5RmFjdG9yeVN0cmF0ZWd5Umlza0xpbWl0Pn0gW3Jpc2tMaW1pdHNdIGFjY291bnQgcmlzayBsaW1pdHMuIFlvdSBjYW4gY29uZmlndXJlIHRyYWRpbmcgdG8gYmVcbiAgICogc3RvcHBlZCBvbmNlIHRvdGFsIGRyYXdkb3duIGdlbmVyYXRlZCBkdXJpbmcgc3BlY2lmaWMgcGVyaW9kIGlzIGV4Y2VlZGVkLiBDYW4gYmUgc3BlY2lmaWVkIGVpdGhlciBmb3IgYmFsYW5jZSBvclxuICAgKiBlcXVpdHkgZHJhd2Rvd25cbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhMZXZlcmFnZV0gc2V0dGluZyBpbmRpY2F0aW5nIG1heHVtdW0gbGV2ZXJhZ2UgYWxsb3dlZCB3aGVuIG9wZW5pbmcgYSBuZXcgcG9zaXRpb25zLlxuICAgKiBBbnkgdHJhZGUgd2hpY2ggcmVzdWx0cyBpbiBhIGhpZ2hlciBsZXZlcmFnZSB3aWxsIGJlIGRpc2NhcmRlZC5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVN0b3BMb3NzXSBmbGFnIGluZGljYXRpbmcgd2hldGhlciBzdG9wIGxvc3Mgc2hvdWxkIGJlIGNvcGllZC4gRGVmYXVsdCBpcyB0byBjb3B5IHN0b3BcbiAgICogbG9zcy5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVRha2VQcm9maXRdIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIHRha2UgcHJvZml0IHNob3VsZCBiZSBjb3BpZWQuIERlZmF1bHQgaXMgdG8gY29weSB0YWtlXG4gICAqIHByb2ZpdC5cbiAgICogQHByb3BlcnR5IHtudW1iZXJ9IFttaW5UcmFkZVZvbHVtZV0gTWluaW11bSB0cmFkZSB2b2x1bWUgdG8gY29weS4gVHJhZGUgc2lnbmFscyB3aXRoIGEgc21hbGxlciB2b2x1bWUgd2lsbCBub3QgYmVcbiAgICogY29waWVkXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbbWF4VHJhZGVWb2x1bWVdIE1heGltdW0gdHJhZGUgdm9sdW1lIHRvIGNvcHkuIFRyYWRlIHNpZ25hbHMgd2l0aCBhIGxhcmdlciB2b2x1bWUgd2lsbCBiZSBjb3BpZWRcbiAgICogd2l0aCBtYXhpbXVtIHZvbHVtZSBpbnN0ZWFkXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8Q29weUZhY3RvcnlTdHJhdGVneVN1YnNjcmlwdGlvbj59IHN1YnNjcmlwdGlvbnMgc3RyYXRlZ3kgc3Vic2NyaXB0aW9uc1xuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3Rvcnkgc3Vic2NyaWJlciBtb2RlbFxuICAgKiBAdHlwZWRlZiB7Q29weUZhY3RvcnlTdWJzY3JpYmVyVXBkYXRlfSBDb3B5RmFjdG9yeVN1YnNjcmliZXJcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IF9pZCBpZCBvZiB0aGUgTWV0YUFwaSBhY2NvdW50IHRvIGNvcHkgdHJhZGVzIHRvIFxuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3RvcnkgcHJvdmlkZXIgc3RyYXRlZ3lcbiAgICogQHR5cGVkZWYge0NvcHlGYWN0b3J5U3RyYXRlZ3lVcGRhdGV9IENvcHlGYWN0b3J5U3RyYXRlZ3lcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IF9pZCB1bmlxdWUgc3RyYXRlZ3kgaWRcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IHBsYXRmb3JtQ29tbWlzc2lvblJhdGUgY29tbWlzc2lvbiByYXRlIHRoZSBwbGF0Zm9ybSBjaGFyZ2VzIGZvciBzdHJhdGVneSBjb3B5aW5nLCBhcHBsaWVkIHRvXG4gICAqIGNvbW1pc3Npb25zIGNoYXJnZWQgYnkgcHJvdmlkZXIuIFRoaXMgY29tbWlzc2lvbiBhcHBsaWVzIG9ubHkgdG8gYWNjb3VudHMgbm90IG1hbmFnZWQgZGlyZWN0bHkgYnkgcHJvdmlkZXIuIFNob3VsZFxuICAgKiBiZSBmcmFjdGlvbiBvZiAxXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBbY2xvc2VPblJlbW92YWxNb2RlXSBwb3NpdGlvbiBjbG9zZSBtb2RlIG9uIHN0cmF0ZWd5IG9yIHN1YnNjcmlwdGlvbiByZW1vdmFsLiBQcmVzZXJ2ZSBtZWFuc1xuICAgKiB0aGF0IHBvc2l0aW9ucyB3aWxsIG5vdCBiZSBjbG9zZWQgYW5kIHdpbGwgbm90IGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnkuIGNsb3NlLWdyYWNlZnVsbHktYnktcG9zaXRpb24gbWVhbnNcbiAgICogdGhhdCBwb3NpdGlvbnMgd2lsbCBjb250aW51ZSB0byBiZSBtYW5hZ2VkIGJ5IENvcHlGYWN0b3J5LCBidXQgb25seSBjbG9zZSBzaWduYWxzIHdpbGwgYmUgY29waWVkLlxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCBtZWFucyB0aGF0IHBvc2l0aW9ucyB3aWxsIGNvbnRpbnVlIHRvIGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnksIGJ1dCBvbmx5IGNsb3NlXG4gICAqIHNpZ25hbHMgd2lsbCBiZSBjb3BpZWQgb3Igc2lnbmFscyB0byBvcGVuIHBvc2l0aW9ucyBmb3IgdGhlIHN5bWJvbHMgd2hpY2ggYWxyZWFkeSBoYXZlIHBvc2l0aW9ucyBvcGVuZWQuXG4gICAqIGNsb3NlLWltbWVkaWF0ZWx5IG1lYW5zIHRoYXQgYWxsIHBvc2l0aW9ucyB3aWxsIGJlIGNsb3NlZCBpbW1lZGlhdGVseS4gRGVmYXVsdCBpcyBjbG9zZS1pbW1lZGlhdGVseS5cbiAgICogVGhpcyBmaWVsZCBjYW4gYmUgY2hhbmdlZCB2aWEgcmVtb3ZlIHBvdGZvbGlvIG1lbWJlciBBUEkgb25seSwgb25lIG9mIHByZXNlcnZlLCBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXBvc2l0aW9uLFxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCwgY2xvc2UtaW1tZWRpYXRlbHlcbiAgICovXG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IHN0cmF0ZWd5IGNvbW1pc3Npb24gc2NoZW1lXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lDb21taXNzaW9uU2NoZW1lXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSB0eXBlIGNvbW1pc3Npb24gdHlwZS4gT25lIG9mIGZsYXQtZmVlLCBsb3RzLXRyYWRlZCwgbG90cy13b24sIGFtb3VudC10cmFkZWQsIGFtb3VudC13b24sXG4gICAqIGhpZ2gtd2F0ZXItbWFya1xuICAgKiBAcHJvcGVydHkge1N0cmluZ30gYmlsbGluZ1BlcmlvZCBiaWxsaW5nIHBlcmlvZC4gT25lIG9mIHdlZWssIG1vbnRoLCBxdWFydGVyXG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBjb21taXNzaW9uUmF0ZSBjb21taXNzaW9uIHJhdGUuIFNob3VsZCBiZSBncmVhdGVyIHRoYW4gb3IgZXF1YWwgdG8gemVybyBpZiBjb21taXNzaW9uIHR5cGUgaXNcbiAgICogZmxhdC1mZWUsIGxvdHMtdHJhZGVkIG9yIGxvdHMtd29uLCBzaG91bGQgYmUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIHplcm8gYW5kIGxlc3MgdGhhbiBvciBlcXVhbCB0byAxIGlmXG4gICAqIGNvbW1pc3Npb24gdHlwZSBpcyBhbW91bnQtdHJhZGVkLCBhbW91bnQtd29uLCBoaWdoLXdhdGVyLW1hcmsuXG4gICAqL1xuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSBzdHJhdGVneSBtYWdpYyBmaWx0ZXJcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlTdHJhdGVneU1hZ2ljRmlsdGVyXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8U3RyaW5nPn0gaW5jbHVkZWQgbGlzdCBvZiBtYWdpY3MgKGV4cGVydCBpZHMpIG9yIG1hZ2ljIHJhbmdlcyBjb3BpZWQuIExlYXZlIHRoZSB2YWx1ZSBlbXB0eSB0b1xuICAgKiBjb3B5IGFsbCBtYWdpY3NcbiAgICogQHByb3BlcnR5IHtBcnJheTxTdHJpbmc+fSBleGNsdWRlZCBsaXN0IG9mIG1hZ2ljcyAoZXhwZXJ0IGlkcykgb3IgbWFnaWMgcmFuZ2VzIGV4Y2x1ZGVkIGZyb20gY29weWluZy4gTGVhdmUgdGhlXG4gICAqIHZhbHVlIGVtcHR5IHRvIGNvcHkgYWxsIG1hZ2ljc1xuICAgKi9cblxuICAvKipcbiAgICogQ29weUZhY3Rvcnkgc3RyYXRlZ3kgdGltZSBzZXR0aW5nc1xuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5VGltZVNldHRpbmdzXG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbbGlmZXRpbWVJbkhvdXJzXSBwb3NpdGlvbiBsaWZldGltZS4gRGVmYXVsdCBpcyB0byBrZWVwIHBvc2l0aW9ucyBvcGVuIHVwIHRvIDkwIGRheXNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFtvcGVuaW5nSW50ZXJ2YWxJbk1pbnV0ZXNdIHRpbWUgaW50ZXJ2YWwgdG8gY29weSBuZXcgcG9zaXRpb25zLiBEZWZhdWx0IGlzIHRvIGxldCAxXG4gICAqIG1pbnV0ZSBmb3IgdGhlIHBvc2l0aW9uIHRvIGdldCBjb3BpZWQuIElmIHBvc2l0aW9uIHdlcmUgbm90IGNvcGllZCBkdXJpbmcgdGhpcyB0aW1lLCB0aGUgY29weWluZyB3aWxsIG5vdCBiZVxuICAgKiByZXRyaWVkIGFueW1vcmUuXG4gICAqL1xuXG4gIC8qKlxuICAgKiBDb3B5RmFjdG9yeSBzdHJhdGVneSBlcXVpdHkgY3VydmUgZmlsdGVyXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lFcXVpdHlDdXJ2ZUZpbHRlclxuICAgKiBAcHJvcGVydHkge051bWJlcn0gcGVyaW9kIG1vdmluZyBhdmVyYWdlIHBlcmlvZCwgbXVzdCBiZSBncmVhdGVyIG9yIGVxdWFsIHRvIDFcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IHRpbWVmcmFtZSBtb3ZpbmcgYXZlcmFnZSB0aW1lZnJhbWUsIGEgcG9zaXRpdmUgaW50ZWdlciBmb2xsb3dlZCBieSB0aW1lIHVuaXQsIGUuZy4gMmguXG4gICAqIEFsbG93ZWQgdW5pdHMgYXJlIHMsIG0sIGgsIGQgYW5kIHcuXG4gICAqL1xuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgQ29weUZhY3RvcnkgY29weSB0cmFkaW5nIHN0cmF0ZWdpZXMuIFNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jb3B5ZmFjdG9yeS9yZXN0QXBpL2FwaS9jb25maWd1cmF0aW9uL2dldFN0cmF0ZWdpZXMvXG4gICAqIEBwYXJhbSB7Qm9vbGVhbn0gW2luY2x1ZGVSZW1vdmVkXSBmbGFnIGluc3RydWN0aW5nIHRvIGluY2x1ZGUgcmVtb3ZlZCBzdHJhdGVnaWVzIGluIHJlc3VsdHNcbiAgICogQHBhcmFtIHtOdW1iZXJ9IFtsaW1pdF0gcGFnaW5hdGlvbiBsaW1pdFxuICAgKiBAcGFyYW0ge051bWJlcn0gW29mZnNldF0gY29weSB0cmFkaW5nIHN0cmF0ZWd5IGlkXG4gICAqIEByZXR1cm4ge1Byb21pc2U8QXJyYXk8Q29weUZhY3RvcnlTdHJhdGVneT4+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIENvcHlGYWN0b3J5IHN0cmF0ZWdpZXMgZm91bmRcbiAgICovXG4gIGdldFN0cmF0ZWdpZXMoaW5jbHVkZVJlbW92ZWQsIGxpbWl0LCBvZmZzZXQpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcignZ2V0U3RyYXRlZ2llcycpO1xuICAgIH1cbiAgICBsZXQgcXMgPSB7fTtcbiAgICBpZihpbmNsdWRlUmVtb3ZlZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5pbmNsdWRlUmVtb3ZlZCA9IGluY2x1ZGVSZW1vdmVkO1xuICAgIH1cbiAgICBpZihsaW1pdCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5saW1pdCA9IGxpbWl0O1xuICAgIH1cbiAgICBpZihvZmZzZXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcXMub2Zmc2V0ID0gb2Zmc2V0O1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vc3RyYXRlZ2llc2AsXG4gICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAgcXMsXG4gICAgICBqc29uOiB0cnVlXG4gICAgfTtcbiAgICByZXR1cm4gdGhpcy5faHR0cENsaWVudC5yZXF1ZXN0KG9wdHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBDb3B5RmFjdG9yeSBjb3B5IHRyYWRpbmcgc3RyYXRlZ3kgYnkgaWQuIFNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jb3B5ZmFjdG9yeS9yZXN0QXBpL2FwaS9jb25maWd1cmF0aW9uL2dldFN0cmF0ZWd5L1xuICAgKiBAcGFyYW0ge3N0cmluZ30gc3RyYXRlZ3kgaWQgdHJhZGluZyBzdHJhdGVneSBpZFxuICAgKiBAcmV0dXJuIHtQcm9taXNlPENvcHlGYWN0b3J5U3RyYXRlZ3k+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIENvcHlGYWN0b3J5IHN0cmF0ZWd5IGZvdW5kXG4gICAqL1xuICBnZXRTdHJhdGVneShzdHJhdGVneUlkKSB7XG4gICAgaWYgKHRoaXMuX2lzTm90Snd0VG9rZW4oKSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2hhbmRsZU5vQWNjZXNzRXJyb3IoJ2dldFN0cmF0ZWd5Jyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvY29uZmlndXJhdGlvbi9zdHJhdGVnaWVzLyR7c3RyYXRlZ3lJZH1gLFxuICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogQ29weUZhY3Rvcnkgc3RyYXRlZ3kgZHJhd2Rvd24gZmlsdGVyXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IENvcHlGYWN0b3J5U3RyYXRlZ3lEcmF3ZG93bkZpbHRlclxuICAgKiBAcHJvcGVydHkge051bWJlcn0gbWF4RHJhd2Rvd24gbWF4aW11bSBkcmF3ZG93biB2YWx1ZSBhZnRlciB3aGljaCBhY3Rpb24gaXMgZXhlY3V0ZWQuIERyYXdkb3duIHNob3VsZFxuICAgKiBiZSBjb25maWd1cmVkIGFzIGEgZnJhY3Rpb24gb2YgMSwgaS5lLiAwLjE1IG1lYW5zIDE1JSBkcmF3ZG93biB2YWx1ZVxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gYWN0aW9uIGFjdGlvbiB0byB0YWtlIHdoZW4gZHJhd2Rvd24gZXhjZWVkcyBtYXhEcmF3ZG93biB2YWx1ZS4gaW5jbHVkZSBtZWFucyB0aGUgdHJhZGluZ1xuICAgKiBzaWduYWwgd2lsbCBiZSB0cmFuc21pdHRlZCBvbmx5IGlmIGRkIGlzIGdyZWF0ZXIgdGhhbiBtYXhEcmF3ZG93biB2YWx1ZS4gZXhjbHVkZSBtZWFucyB0aGUgdHJhZGluZyBzaWduYWwgXG4gICAqIHdpbGwgYmUgdHJhbnNtaXR0ZWQgb25seSBpZiBkZCBpcyBsZXNzIHRoYW4gbWF4RHJhd2Rvd24gdmFsdWVcbiAgICovXG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IHN0cmF0ZWd5IHVwZGF0ZVxuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeVN0cmF0ZWd5VXBkYXRlXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBuYW1lIHN0cmF0ZWd5IGh1bWFuLXJlYWRhYmxlIG5hbWVcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IGRlc2NyaXB0aW9uIGxvbmdlciBzdHJhdGVneSBodW1hbi1yZWFkYWJsZSBkZXNjcmlwdGlvblxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gYWNjb3VudElkIGlkIG9mIHRoZSBNZXRhQXBpIGFjY291bnQgcHJvdmlkaW5nIHRoZSBzdHJhdGVneVxuICAgKiBAcHJvcGVydHkge0Jvb2xlYW59IFtza2lwUGVuZGluZ09yZGVyc10gZmxhZyBpbmRpY2F0aW5nIHRoYXQgcGVuZGluZyBvcmRlcnMgc2hvdWxkIG5vdCBiZSBjb3BpZWQuXG4gICAqIERlZmF1bHQgaXMgdG8gY29weSBwZW5kaW5nIG9yZGVyc1xuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lDb21taXNzaW9uU2NoZW1lfSBbY29tbWlzc2lvblNjaGVtZV0gY29tbWlzc2lvbiBzY2hlbWUgYWxsb3dlZCBieSB0aGlzIHN0cmF0ZWd5XG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbbWF4VHJhZGVSaXNrXSBtYXggcmlzayBwZXIgdHJhZGUsIGV4cHJlc3NlZCBhcyBhIGZyYWN0aW9uIG9mIDEuIElmIHRyYWRlIGhhcyBhIFNMLCB0aGVcbiAgICogdHJhZGUgc2l6ZSB3aWxsIGJlIGFkanVzdGVkIHRvIG1hdGNoIHRoZSByaXNrIGxpbWl0LiBJZiBub3QsIHRoZSB0cmFkZSBTTCB3aWxsIGJlIGFwcGxpZWQgYWNjb3JkaW5nIHRvIHRoZSByaXNrXG4gICAqIGxpbWl0XG4gICAqIEBwcm9wZXJ0eSB7Qm9vbGVhbn0gW3JldmVyc2VdIGZsYWcgaW5kaWNhdGluZyB0aGF0IHRoZSBzdHJhdGVneSBzaG91bGQgYmUgY29waWVkIGluIGEgcmV2ZXJzZSBkaXJlY3Rpb25cbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IFtyZWR1Y2VDb3JyZWxhdGlvbnNdIHNldHRpbmcgaW5kaWNhdGluZyB3aGV0aGVyIHRvIGVuYWJsZSBhdXRvbWF0aWMgdHJhZGVcbiAgICogY29ycmVsYXRpb24gcmVkdWN0aW9uLiBQb3NzaWJsZSBzZXR0aW5ncyBhcmUgbm90IHNwZWNpZmllZCAoZGlzYWJsZSBjb3JyZWxhdGlvbiByaXNrIHJlc3RyaWN0aW9ucyksXG4gICAqIGJ5LXN0cmF0ZWd5IChsaW1pdCBjb3JyZWxhdGlvbnMgZm9yIHRoZSBzdHJhdGVneSkgb3IgYnktYWNjb3VudCAobGltaXQgY29ycmVsYXRpb25zIGZvciB0aGUgYWNjb3VudClcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5U3RvcE91dFNldHRpbmdzfSBbc3RvcE91dFJpc2tdIHN0b3Agb3V0IHNldHRpbmcuIEFsbCB0cmFkaW5nIHdpbGxcbiAgICogYmUgdGVybWluYXRlZCBhbmQgcG9zaXRpb25zIGNsb3NlZCBvbmNlIGVxdWl0eSBkcmF3ZG93biByZWFjaGVzIHRoaXMgdmFsdWVcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5U3ltYm9sRmlsdGVyfSBbc3ltYm9sRmlsdGVyXSBzeW1ib2wgZmlsdGVycyB3aGljaCBjYW4gYmUgdXNlZCB0byBjb3B5IG9ubHkgc3BlY2lmaWNcbiAgICogc3ltYm9scyBvciBleGNsdWRlIHNvbWUgc3ltYm9scyBmcm9tIGNvcHlpbmdcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5TmV3c0ZpbHRlcn0gW25ld3NGaWx0ZXJdIG5ld3MgcmlzayBmaWx0ZXIgY29uZmlndXJhdGlvblxuICAgKiBAcHJvcGVydHkge0FycmF5PENvcHlGYWN0b3J5U3RyYXRlZ3lSaXNrTGltaXQ+fSBbcmlza0xpbWl0c10gc3RyYXRlZ3kgcmlzayBsaW1pdHMuIFlvdSBjYW4gY29uZmlndXJlXG4gICAqIHRyYWRpbmcgdG8gYmUgc3RvcHBlZCBvbmNlIHRvdGFsIGRyYXdkb3duIGdlbmVyYXRlZCBkdXJpbmcgc3BlY2lmaWMgcGVyaW9kIGlzIGV4Y2VlZGVkLiBDYW4gYmUgc3BlY2lmaWVkIGVpdGhlciBmb3JcbiAgICogYmFsYW5jZSBvciBlcXVpdHkgZHJhd2Rvd25cbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5TWF4U3RvcExvc3N9IFttYXhTdG9wTG9zc10gc3RvcCBsb3NzIHZhbHVlIHJlc3RyaWN0aW9uXG4gICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBbbWF4TGV2ZXJhZ2VdIG1heCBsZXZlcmFnZSByaXNrIHJlc3RyaWN0aW9uLiBBbGwgdHJhZGVzIHJlc3VsdGluZyBpbiBhIGxldmVyYWdlIHZhbHVlXG4gICAqIGhpZ2hlciB0aGFuIHNwZWNpZmllZCB3aWxsIGJlIHNraXBwZWRcbiAgICogQHByb3BlcnR5IHtBcnJheTxDb3B5RmFjdG9yeVN0cmF0ZWd5U3ltYm9sTWFwcGluZz59IFtzeW1ib2xNYXBwaW5nXSBkZWZpbmVzIGhvdyBzeW1ib2wgbmFtZSBzaG91bGQgYmUgY2hhbmdlZCB3aGVuXG4gICAqIHRyYWRpbmcgKGUuZy4gd2hlbiBicm9rZXIgdXNlcyBzeW1ib2wgbmFtZXMgd2l0aCB1bnVzdWFsIHN1ZmZpeGVzKS4gQnkgZGVmYXVsdCB0aGlzIHNldHRpbmcgaXMgZGlzYWJsZWQgYW5kIHRoZVxuICAgKiB0cmFkZXMgYXJlIGNvcGllZCB1c2luZyBzaWduYWwgc291cmNlIHN5bWJvbCBuYW1lXG4gICAqIEBwcm9wZXJ0eSB7Q29weUZhY3RvcnlTdHJhdGVneVRyYWRlU2l6ZVNjYWxpbmd9IFt0cmFkZVNpemVTY2FsaW5nXSBUcmFkZSBzaXplIHNjYWxpbmcgc2V0dGluZ3MuIEJ5IGRlZmF1bHQgdGhlXG4gICAqIHRyYWRlIHNpemUgb24gc3RyYXRlZ3kgc3Vic2NyaWJlciBzaWRlIHdpbGwgYmUgc2NhbGVkIGFjY29yZGluZyB0byBiYWxhbmNlIHRvIHByZXNlcnZlIHJpc2suXG4gICAqIEBwcm9wZXJ0eSB7Ym9vbGVhbn0gW2NvcHlTdG9wTG9zc10gZmxhZyBpbmRpY2F0aW5nIHdoZXRoZXIgc3RvcCBsb3NzIHNob3VsZCBiZSBjb3BpZWQuIERlZmF1bHQgaXMgdG8gY29weSBzdG9wXG4gICAqIGxvc3MuXG4gICAqIEBwcm9wZXJ0eSB7Ym9vbGVhbn0gW2NvcHlUYWtlUHJvZml0XSBmbGFnIGluZGljYXRpbmcgd2hldGhlciB0YWtlIHByb2ZpdCBzaG91bGQgYmUgY29waWVkLiBEZWZhdWx0IGlzIHRvIGNvcHkgdGFrZVxuICAgKiBwcm9maXQuXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbbWluVHJhZGVWb2x1bWVdIE1pbmltdW0gdHJhZGUgdm9sdW1lIHRvIGNvcHkuIFRyYWRlIHNpZ25hbHMgd2l0aCBhIHNtYWxsZXIgdm9sdW1lIHdpbGwgbm90IGJlXG4gICAqIGNvcGllZFxuICAgKiBAcHJvcGVydHkge251bWJlcn0gW21heFRyYWRlVm9sdW1lXSBNYXhpbXVtIHRyYWRlIHZvbHVtZSB0byBjb3B5LiBUcmFkZSBzaWduYWxzIHdpdGggYSBsYXJnZXIgdm9sdW1lIHdpbGwgYmUgY29waWVkXG4gICAqIHdpdGggbWF4aW11bSB2b2x1bWUgaW5zdGVhZFxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lNYWdpY0ZpbHRlcn0gW21hZ2ljRmlsdGVyXSBtYWdpYyAoZXhwZXJ0IGlkKSBmaWx0ZXJcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5RXF1aXR5Q3VydmVGaWx0ZXJ9IFtlcXVpdHlDdXJ2ZUZpbHRlcl0gZmlsdGVyIHdoaWNoIHBlcm1pdHMgdGhlIHRyYWRlcyBvbmx5IGlmIGFjY291bnRcbiAgICogZXF1aXR5IGlzIGdyZWF0ZXIgdGhhbiBiYWxhbmNlIG1vdmluZyBhdmVyYWdlXG4gICAqIEBwcm9wZXJ0eSB7Q29weUZhY3RvcnlTdHJhdGVneURyYXdkb3duRmlsdGVyfSBbZHJhd2Rvd25GaWx0ZXJdIG1hc3RlciBhY2NvdW50IHN0cmF0ZWd5IGRyYXdkb3duIGZpbHRlclxuICAgKiBAcHJvcGVydHkge0FycmF5PFN0cmluZz59IFtzeW1ib2xzVHJhZGVkXSBzeW1ib2xzIHRyYWRlZCBieSB0aGlzIHN0cmF0ZWd5LiBTcGVjaWZ5aW5nIHRoZSBzeW1ib2xzIHdpbGwgaW1wcm92ZSB0cmFkZVxuICAgKiBsYXRlbmN5IG9uIGZpcnN0IHRyYWRlcyBtYWRlIGJ5IHN0cmF0ZWd5LiBJZiB5b3UgZG8gbm90IHNwZWNpZnkgdGhpcyBzZXR0aW5nIHRoZSBhcHBsaWNhdGlvbiB3aWxsIG1vbml0b3IgdGhlIHN0cmF0ZWd5XG4gICAqIHRyYWRlcyBhbmQgZGV0ZWN0IHRoZSBzeW1ib2xzIGF1dG9tYXRpY2FsbHkgb3ZlciB0aW1lIFxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lUaW1lU2V0dGluZ3N9IFt0aW1lU2V0dGluZ3NdIHNldHRpbmdzIHRvIG1hbmFnZSBjb3B5aW5nIHRpbWVmcmFtZSBhbmQgcG9zaXRpb25cbiAgICogbGlmZXRpbWUuIERlZmF1bHQgaXMgdG8gY29weSBwb3NpdGlvbiB3aXRoaW4gMSBtaW51dGUgZnJvbSBiZWluZyBvcGVuZWQgYXQgc291cmNlIGFuZCBsZXQgdGhlIHBvc2l0aW9uIHRvIGxpdmUgZm9yXG4gICAqIHVwIHRvIDkwIGRheXNcbiAgICovXG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBDb3B5RmFjdG9yeSBzdHJhdGVneS4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vdXBkYXRlU3RyYXRlZ3kvXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBzdHJhdGVneUlkIGNvcHkgdHJhZGluZyBzdHJhdGVneSBpZFxuICAgKiBAcGFyYW0ge0NvcHlGYWN0b3J5U3RyYXRlZ3lVcGRhdGV9IHN0cmF0ZWd5IHRyYWRpbmcgc3RyYXRlZ3kgdXBkYXRlXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gc3RyYXRlZ3kgaXMgdXBkYXRlZFxuICAgKi9cbiAgdXBkYXRlU3RyYXRlZ3koc3RyYXRlZ3lJZCwgc3RyYXRlZ3kpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcigndXBkYXRlU3RyYXRlZ3knKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIHVybDogYCR7dGhpcy5faG9zdH0vdXNlcnMvY3VycmVudC9jb25maWd1cmF0aW9uL3N0cmF0ZWdpZXMvJHtzdHJhdGVneUlkfWAsXG4gICAgICBtZXRob2Q6ICdQVVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAgYm9keTogc3RyYXRlZ3ksXG4gICAgICBqc29uOiB0cnVlXG4gICAgfTtcbiAgICByZXR1cm4gdGhpcy5faHR0cENsaWVudC5yZXF1ZXN0KG9wdHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvcHlGYWN0b3J5IGNsb3NlIGluc3RydWN0aW9uc1xuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb3B5RmFjdG9yeUNsb3NlSW5zdHJ1Y3Rpb25zXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBbbW9kZV0gcG9zaXRpb24gY2xvc2UgbW9kZSBvbiBzdHJhdGVneSBvciBzdWJzY3JpcHRpb24gcmVtb3ZhbC4gUHJlc2VydmUgbWVhbnNcbiAgICogdGhhdCBwb3NpdGlvbnMgd2lsbCBub3QgYmUgY2xvc2VkIGFuZCB3aWxsIG5vdCBiZSBtYW5hZ2VkIGJ5IENvcHlGYWN0b3J5LlxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXBvc2l0aW9uIG1lYW5zIHRoYXQgcG9zaXRpb25zIHdpbGwgY29udGludWUgdG8gYmUgbWFuYWdlZCBieSBDb3B5RmFjdG9yeSxcbiAgICogYnV0IG9ubHkgY2xvc2Ugc2lnbmFscyB3aWxsIGJlIGNvcGllZC4gY2xvc2UtZ3JhY2VmdWxseS1ieS1zeW1ib2wgbWVhbnMgdGhhdCBwb3NpdGlvbnMgd2lsbFxuICAgKiBjb250aW51ZSB0byBiZSBtYW5hZ2VkIGJ5IENvcHlGYWN0b3J5LCBidXQgb25seSBjbG9zZSBzaWduYWxzIHdpbGwgYmUgY29waWVkIG9yIHNpZ25hbHMgdG9cbiAgICogb3BlbiBwb3NpdGlvbnMgZm9yIHRoZSBzeW1ib2xzIHdoaWNoIGFscmVhZHkgaGF2ZSBwb3NpdGlvbnMgb3BlbmVkLiBjbG9zZS1pbW1lZGlhdGVseSBtZWFuc1xuICAgKiB0aGF0IGFsbCBwb3NpdGlvbnMgd2lsbCBiZSBjbG9zZWQgaW1tZWRpYXRlbHkuIERlZmF1bHQgaXMgY2xvc2UtaW1tZWRpYXRlbHkuIE9uZSBvZiAncHJlc2VydmUnLFxuICAgKiAnY2xvc2UtZ3JhY2VmdWxseS1ieS1wb3NpdGlvbicsICdjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCcsICdjbG9zZS1pbW1lZGlhdGVseSdcbiAgICogQHByb3BlcnR5IHtEYXRlfSBbcmVtb3ZlQWZ0ZXJdIHRpbWUgdG8gZm9yY2UgcmVtb3ZlIG9iamVjdCBhZnRlci4gVGhlIG9iamVjdCB3aWxsIGJlIHJlbW92ZWQgYWZ0ZXJcbiAgICogdGhpcyB0aW1lLCBldmVuIGlmIHBvc2l0aW9ucyBhcmUgbm90IHlldCBjbG9zZWQgZnVsbHkuIERlZmF1bHQgaXMgY3VycmVudCBkYXRlIHBsdXMgMzAgZGF5cy5cbiAgICogQ2FuIG5vdCBiZSBsZXNzIHRoYW4gMzAgZGF5cyBvciBncmVhdGVyIHRoYW4gY3VycmVudCBkYXRlIHBsdXMgOTAgZGF5cy4gVGhlIHNldHRpbmcgaXMgaWdub3JlZFxuICAgKiB3aGVuIGEgc3Vic2NyaXB0aW9uIGlzIGJlaW5nIHJlbW92ZWRcbiAgICovXG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBDb3B5RmFjdG9yeSBzdHJhdGVneS4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vcmVtb3ZlU3RyYXRlZ3kvXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBzdHJhdGVneUlkIGNvcHkgdHJhZGluZyBzdHJhdGVneSBpZFxuICAgKiBAcGFyYW0ge0NvcHlGYWN0b3J5Q2xvc2VJbnN0cnVjdGlvbnN9IFtjbG9zZUluc3RydWN0aW9uc10gc3RyYXRlZ3kgY2xvc2UgaW5zdHJ1Y3Rpb25zXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gc3RyYXRlZ3kgaXMgcmVtb3ZlZFxuICAgKi9cbiAgcmVtb3ZlU3RyYXRlZ3koc3RyYXRlZ3lJZCwgY2xvc2VJbnN0cnVjdGlvbnMpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcigncmVtb3ZlU3RyYXRlZ3knKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIHVybDogYCR7dGhpcy5faG9zdH0vdXNlcnMvY3VycmVudC9jb25maWd1cmF0aW9uL3N0cmF0ZWdpZXMvJHtzdHJhdGVneUlkfWAsXG4gICAgICBtZXRob2Q6ICdERUxFVEUnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAgYm9keTogY2xvc2VJbnN0cnVjdGlvbnMsXG4gICAgICBqc29uOiB0cnVlXG4gICAgfTtcbiAgICByZXR1cm4gdGhpcy5faHR0cENsaWVudC5yZXF1ZXN0KG9wdHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFBvcnRmb2xpbyBzdHJhdGVneSBtZW1iZXJcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlQb3J0Zm9saW9TdHJhdGVneU1lbWJlclxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gc3RyYXRlZ3lJZCBtZW1iZXIgc3RyYXRlZ3kgaWRcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IG11bHRpcGxpZXIgY29weWluZyBtdWx0aXBsaWVyICh3ZWlnaHQgaW4gdGhlIHBvcnRmb2xpbylcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBbc2tpcFBlbmRpbmdPcmRlcnNdIGZsYWcgaW5kaWNhdGluZyB0aGF0IHBlbmRpbmcgb3JkZXJzIHNob3VsZCBub3QgYmUgY29waWVkLlxuICAgKiBEZWZhdWx0IGlzIHRvIGNvcHkgcGVuZGluZyBvcmRlcnNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhUcmFkZVJpc2tdIG1heCByaXNrIHBlciB0cmFkZSwgZXhwcmVzc2VkIGFzIGEgZnJhY3Rpb24gb2YgMS4gSWYgdHJhZGUgaGFzIGEgU0wsIHRoZVxuICAgKiB0cmFkZSBzaXplIHdpbGwgYmUgYWRqdXN0ZWQgdG8gbWF0Y2ggdGhlIHJpc2sgbGltaXQuIElmIG5vdCwgdGhlIHRyYWRlIFNMIHdpbGwgYmUgYXBwbGllZCBhY2NvcmRpbmcgdG8gdGhlIHJpc2tcbiAgICogbGltaXRcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBbcmV2ZXJzZV0gZmxhZyBpbmRpY2F0aW5nIHRoYXQgdGhlIHN0cmF0ZWd5IHNob3VsZCBiZSBjb3BpZWQgaW4gYSByZXZlcnNlIGRpcmVjdGlvblxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gW3JlZHVjZUNvcnJlbGF0aW9uc10gc2V0dGluZyBpbmRpY2F0aW5nIHdoZXRoZXIgdG8gZW5hYmxlIGF1dG9tYXRpYyB0cmFkZVxuICAgKiBjb3JyZWxhdGlvbiByZWR1Y3Rpb24uIFBvc3NpYmxlIHNldHRpbmdzIGFyZSBub3Qgc3BlY2lmaWVkIChkaXNhYmxlIGNvcnJlbGF0aW9uIHJpc2sgcmVzdHJpY3Rpb25zKSxcbiAgICogYnktc3RyYXRlZ3kgKGxpbWl0IGNvcnJlbGF0aW9ucyBmb3IgdGhlIHN0cmF0ZWd5KSBvciBieS1hY2NvdW50IChsaW1pdCBjb3JyZWxhdGlvbnMgZm9yIHRoZSBhY2NvdW50KVxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lTdG9wT3V0U2V0dGluZ3N9IFtzdG9wT3V0Umlza10gc3RvcCBvdXQgc2V0dGluZy4gQWxsIHRyYWRpbmcgd2lsbFxuICAgKiBiZSB0ZXJtaW5hdGVkIGFuZCBwb3NpdGlvbnMgY2xvc2VkIG9uY2UgZXF1aXR5IGRyYXdkb3duIHJlYWNoZXMgdGhpcyB2YWx1ZVxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xGaWx0ZXJ9IFtzeW1ib2xGaWx0ZXJdIHN5bWJvbCBmaWx0ZXJzIHdoaWNoIGNhbiBiZSB1c2VkIHRvIGNvcHkgb25seSBzcGVjaWZpY1xuICAgKiBzeW1ib2xzIG9yIGV4Y2x1ZGUgc29tZSBzeW1ib2xzIGZyb20gY29weWluZ1xuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lOZXdzRmlsdGVyfSBbbmV3c0ZpbHRlcl0gbmV3cyByaXNrIGZpbHRlciBjb25maWd1cmF0aW9uXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8Q29weUZhY3RvcnlTdHJhdGVneVJpc2tMaW1pdD59IFtyaXNrTGltaXRzXSBzdHJhdGVneSByaXNrIGxpbWl0cy4gWW91IGNhbiBjb25maWd1cmVcbiAgICogdHJhZGluZyB0byBiZSBzdG9wcGVkIG9uY2UgdG90YWwgZHJhd2Rvd24gZ2VuZXJhdGVkIGR1cmluZyBzcGVjaWZpYyBwZXJpb2QgaXMgZXhjZWVkZWQuIENhbiBiZSBzcGVjaWZpZWQgZWl0aGVyIGZvclxuICAgKiBiYWxhbmNlIG9yIGVxdWl0eSBkcmF3ZG93blxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lNYXhTdG9wTG9zc30gW21heFN0b3BMb3NzXSBzdG9wIGxvc3MgdmFsdWUgcmVzdHJpY3Rpb25cbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhMZXZlcmFnZV0gbWF4IGxldmVyYWdlIHJpc2sgcmVzdHJpY3Rpb24uIEFsbCB0cmFkZXMgcmVzdWx0aW5nIGluIGEgbGV2ZXJhZ2UgdmFsdWVcbiAgICogaGlnaGVyIHRoYW4gc3BlY2lmaWVkIHdpbGwgYmUgc2tpcHBlZFxuICAgKiBAcHJvcGVydHkge0FycmF5PENvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xNYXBwaW5nPn0gW3N5bWJvbE1hcHBpbmddIGRlZmluZXMgaG93IHN5bWJvbCBuYW1lIHNob3VsZCBiZSBjaGFuZ2VkIHdoZW5cbiAgICogdHJhZGluZyAoZS5nLiB3aGVuIGJyb2tlciB1c2VzIHN5bWJvbCBuYW1lcyB3aXRoIHVudXN1YWwgc3VmZml4ZXMpLiBCeSBkZWZhdWx0IHRoaXMgc2V0dGluZyBpcyBkaXNhYmxlZCBhbmQgdGhlXG4gICAqIHRyYWRlcyBhcmUgY29waWVkIHVzaW5nIHNpZ25hbCBzb3VyY2Ugc3ltYm9sIG5hbWVcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5VHJhZGVTaXplU2NhbGluZ30gW3RyYWRlU2l6ZVNjYWxpbmddIFRyYWRlIHNpemUgc2NhbGluZyBzZXR0aW5ncy4gQnkgZGVmYXVsdCB0aGVcbiAgICogdHJhZGUgc2l6ZSBvbiBzdHJhdGVneSBzdWJzY3JpYmVyIHNpZGUgd2lsbCBiZSBzY2FsZWQgYWNjb3JkaW5nIHRvIGJhbGFuY2UgdG8gcHJlc2VydmUgcmlzay5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVN0b3BMb3NzXSBmbGFnIGluZGljYXRpbmcgd2hldGhlciBzdG9wIGxvc3Mgc2hvdWxkIGJlIGNvcGllZC4gRGVmYXVsdCBpcyB0byBjb3B5IHN0b3BcbiAgICogbG9zcy5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVRha2VQcm9maXRdIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIHRha2UgcHJvZml0IHNob3VsZCBiZSBjb3BpZWQuIERlZmF1bHQgaXMgdG8gY29weSB0YWtlXG4gICAqIHByb2ZpdC5cbiAgICogQHByb3BlcnR5IHtudW1iZXJ9IFttaW5UcmFkZVZvbHVtZV0gTWluaW11bSB0cmFkZSB2b2x1bWUgdG8gY29weS4gVHJhZGUgc2lnbmFscyB3aXRoIGEgc21hbGxlciB2b2x1bWUgd2lsbCBub3QgYmVcbiAgICogY29waWVkXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbbWF4VHJhZGVWb2x1bWVdIE1heGltdW0gdHJhZGUgdm9sdW1lIHRvIGNvcHkuIFRyYWRlIHNpZ25hbHMgd2l0aCBhIGxhcmdlciB2b2x1bWUgd2lsbCBiZSBjb3BpZWRcbiAgICogd2l0aCBtYXhpbXVtIHZvbHVtZSBpbnN0ZWFkXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBbY2xvc2VPblJlbW92YWxNb2RlXSBwb3NpdGlvbiBjbG9zZSBtb2RlIG9uIHN0cmF0ZWd5IG9yIHN1YnNjcmlwdGlvbiByZW1vdmFsLiBQcmVzZXJ2ZSBtZWFuc1xuICAgKiB0aGF0IHBvc2l0aW9ucyB3aWxsIG5vdCBiZSBjbG9zZWQgYW5kIHdpbGwgbm90IGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnkuIGNsb3NlLWdyYWNlZnVsbHktYnktcG9zaXRpb24gbWVhbnNcbiAgICogdGhhdCBwb3NpdGlvbnMgd2lsbCBjb250aW51ZSB0byBiZSBtYW5hZ2VkIGJ5IENvcHlGYWN0b3J5LCBidXQgb25seSBjbG9zZSBzaWduYWxzIHdpbGwgYmUgY29waWVkLlxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCBtZWFucyB0aGF0IHBvc2l0aW9ucyB3aWxsIGNvbnRpbnVlIHRvIGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnksIGJ1dCBvbmx5IGNsb3NlXG4gICAqIHNpZ25hbHMgd2lsbCBiZSBjb3BpZWQgb3Igc2lnbmFscyB0byBvcGVuIHBvc2l0aW9ucyBmb3IgdGhlIHN5bWJvbHMgd2hpY2ggYWxyZWFkeSBoYXZlIHBvc2l0aW9ucyBvcGVuZWQuXG4gICAqIGNsb3NlLWltbWVkaWF0ZWx5IG1lYW5zIHRoYXQgYWxsIHBvc2l0aW9ucyB3aWxsIGJlIGNsb3NlZCBpbW1lZGlhdGVseS4gRGVmYXVsdCBpcyBjbG9zZS1pbW1lZGlhdGVseS5cbiAgICogVGhpcyBmaWVsZCBjYW4gYmUgY2hhbmdlZCB2aWEgcmVtb3ZlIHBvdGZvbGlvIG1lbWJlciBBUEkgb25seSwgb25lIG9mIHByZXNlcnZlLCBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXBvc2l0aW9uLFxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCwgY2xvc2UtaW1tZWRpYXRlbHlcbiAgICovXG5cbiAgLyoqXG4gICAqIFBvcnRmb2xpbyBzdHJhdGVneSB1cGRhdGVcbiAgICogQHR5cGVkZWYge09iamVjdH0gQ29weUZhY3RvcnlQb3J0Zm9saW9TdHJhdGVneVVwZGF0ZVxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gbmFtZSBzdHJhdGVneSBodW1hbi1yZWFkYWJsZSBuYW1lXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBkZXNjcmlwdGlvbiBsb25nZXIgc3RyYXRlZ3kgaHVtYW4tcmVhZGFibGUgZGVzY3JpcHRpb25cbiAgICogQHByb3BlcnR5IHtBcnJheTxDb3B5RmFjdG9yeVBvcnRmb2xpb1N0cmF0ZWd5TWVtYmVyPn0gbWVtYmVycyBhcnJheSBvZiBwb3J0Zm9saW8gbWVtYmVyc1xuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lDb21taXNzaW9uU2NoZW1lfSBbY29tbWlzc2lvblNjaGVtZV0gY29tbWlzc2lvbiBzY2hlbWUgYWxsb3dlZCBieSB0aGlzIHN0cmF0ZWd5LiBCeVxuICAgKiBkZWZhdWx0IG1vbnRobHkgYmlsbGluZyBwZXJpb2Qgd2l0aCBubyBjb21taXNzaW9uIGlzIGJlaW5nIHVzZWRcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBbc2tpcFBlbmRpbmdPcmRlcnNdIGZsYWcgaW5kaWNhdGluZyB0aGF0IHBlbmRpbmcgb3JkZXJzIHNob3VsZCBub3QgYmUgY29waWVkLlxuICAgKiBEZWZhdWx0IGlzIHRvIGNvcHkgcGVuZGluZyBvcmRlcnNcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhUcmFkZVJpc2tdIG1heCByaXNrIHBlciB0cmFkZSwgZXhwcmVzc2VkIGFzIGEgZnJhY3Rpb24gb2YgMS4gSWYgdHJhZGUgaGFzIGEgU0wsIHRoZVxuICAgKiB0cmFkZSBzaXplIHdpbGwgYmUgYWRqdXN0ZWQgdG8gbWF0Y2ggdGhlIHJpc2sgbGltaXQuIElmIG5vdCwgdGhlIHRyYWRlIFNMIHdpbGwgYmUgYXBwbGllZCBhY2NvcmRpbmcgdG8gdGhlIHJpc2tcbiAgICogbGltaXRcbiAgICogQHByb3BlcnR5IHtCb29sZWFufSBbcmV2ZXJzZV0gZmxhZyBpbmRpY2F0aW5nIHRoYXQgdGhlIHN0cmF0ZWd5IHNob3VsZCBiZSBjb3BpZWQgaW4gYSByZXZlcnNlIGRpcmVjdGlvblxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gW3JlZHVjZUNvcnJlbGF0aW9uc10gc2V0dGluZyBpbmRpY2F0aW5nIHdoZXRoZXIgdG8gZW5hYmxlIGF1dG9tYXRpYyB0cmFkZVxuICAgKiBjb3JyZWxhdGlvbiByZWR1Y3Rpb24uIFBvc3NpYmxlIHNldHRpbmdzIGFyZSBub3Qgc3BlY2lmaWVkIChkaXNhYmxlIGNvcnJlbGF0aW9uIHJpc2sgcmVzdHJpY3Rpb25zKSxcbiAgICogYnktc3RyYXRlZ3kgKGxpbWl0IGNvcnJlbGF0aW9ucyBmb3IgdGhlIHN0cmF0ZWd5KSBvciBieS1hY2NvdW50IChsaW1pdCBjb3JyZWxhdGlvbnMgZm9yIHRoZSBhY2NvdW50KVxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lTdG9wT3V0U2V0dGluZ3N9IFtzdG9wT3V0Umlza10gc3RvcCBvdXQgc2V0dGluZy4gQWxsIHRyYWRpbmcgd2lsbFxuICAgKiBiZSB0ZXJtaW5hdGVkIGFuZCBwb3NpdGlvbnMgY2xvc2VkIG9uY2UgZXF1aXR5IGRyYXdkb3duIHJlYWNoZXMgdGhpcyB2YWx1ZVxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xGaWx0ZXJ9IFtzeW1ib2xGaWx0ZXJdIHN5bWJvbCBmaWx0ZXJzIHdoaWNoIGNhbiBiZSB1c2VkIHRvIGNvcHkgb25seSBzcGVjaWZpY1xuICAgKiBzeW1ib2xzIG9yIGV4Y2x1ZGUgc29tZSBzeW1ib2xzIGZyb20gY29weWluZ1xuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lOZXdzRmlsdGVyfSBbbmV3c0ZpbHRlcl0gbmV3cyByaXNrIGZpbHRlciBjb25maWd1cmF0aW9uXG4gICAqIEBwcm9wZXJ0eSB7QXJyYXk8Q29weUZhY3RvcnlTdHJhdGVneVJpc2tMaW1pdD59IFtyaXNrTGltaXRzXSBzdHJhdGVneSByaXNrIGxpbWl0cy4gWW91IGNhbiBjb25maWd1cmVcbiAgICogdHJhZGluZyB0byBiZSBzdG9wcGVkIG9uY2UgdG90YWwgZHJhd2Rvd24gZ2VuZXJhdGVkIGR1cmluZyBzcGVjaWZpYyBwZXJpb2QgaXMgZXhjZWVkZWQuIENhbiBiZSBzcGVjaWZpZWQgZWl0aGVyIGZvclxuICAgKiBiYWxhbmNlIG9yIGVxdWl0eSBkcmF3ZG93blxuICAgKiBAcHJvcGVydHkge0NvcHlGYWN0b3J5U3RyYXRlZ3lNYXhTdG9wTG9zc30gW21heFN0b3BMb3NzXSBzdG9wIGxvc3MgdmFsdWUgcmVzdHJpY3Rpb25cbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IFttYXhMZXZlcmFnZV0gbWF4IGxldmVyYWdlIHJpc2sgcmVzdHJpY3Rpb24uIEFsbCB0cmFkZXMgcmVzdWx0aW5nIGluIGEgbGV2ZXJhZ2UgdmFsdWVcbiAgICogaGlnaGVyIHRoYW4gc3BlY2lmaWVkIHdpbGwgYmUgc2tpcHBlZFxuICAgKiBAcHJvcGVydHkge0FycmF5PENvcHlGYWN0b3J5U3RyYXRlZ3lTeW1ib2xNYXBwaW5nPn0gW3N5bWJvbE1hcHBpbmddIGRlZmluZXMgaG93IHN5bWJvbCBuYW1lIHNob3VsZCBiZSBjaGFuZ2VkIHdoZW5cbiAgICogdHJhZGluZyAoZS5nLiB3aGVuIGJyb2tlciB1c2VzIHN5bWJvbCBuYW1lcyB3aXRoIHVudXN1YWwgc3VmZml4ZXMpLiBCeSBkZWZhdWx0IHRoaXMgc2V0dGluZyBpcyBkaXNhYmxlZCBhbmQgdGhlXG4gICAqIHRyYWRlcyBhcmUgY29waWVkIHVzaW5nIHNpZ25hbCBzb3VyY2Ugc3ltYm9sIG5hbWVcbiAgICogQHByb3BlcnR5IHtDb3B5RmFjdG9yeVN0cmF0ZWd5VHJhZGVTaXplU2NhbGluZ30gW3RyYWRlU2l6ZVNjYWxpbmddIFRyYWRlIHNpemUgc2NhbGluZyBzZXR0aW5ncy4gQnkgZGVmYXVsdCB0aGVcbiAgICogdHJhZGUgc2l6ZSBvbiBzdHJhdGVneSBzdWJzY3JpYmVyIHNpZGUgd2lsbCBiZSBzY2FsZWQgYWNjb3JkaW5nIHRvIGJhbGFuY2UgdG8gcHJlc2VydmUgcmlzay5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVN0b3BMb3NzXSBmbGFnIGluZGljYXRpbmcgd2hldGhlciBzdG9wIGxvc3Mgc2hvdWxkIGJlIGNvcGllZC4gRGVmYXVsdCBpcyB0byBjb3B5IHN0b3BcbiAgICogbG9zcy5cbiAgICogQHByb3BlcnR5IHtib29sZWFufSBbY29weVRha2VQcm9maXRdIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIHRha2UgcHJvZml0IHNob3VsZCBiZSBjb3BpZWQuIERlZmF1bHQgaXMgdG8gY29weSB0YWtlXG4gICAqIHByb2ZpdC5cbiAgICogQHByb3BlcnR5IHtudW1iZXJ9IFttaW5UcmFkZVZvbHVtZV0gTWluaW11bSB0cmFkZSB2b2x1bWUgdG8gY29weS4gVHJhZGUgc2lnbmFscyB3aXRoIGEgc21hbGxlciB2b2x1bWUgd2lsbCBub3QgYmVcbiAgICogY29waWVkXG4gICAqIEBwcm9wZXJ0eSB7bnVtYmVyfSBbbWF4VHJhZGVWb2x1bWVdIE1heGltdW0gdHJhZGUgdm9sdW1lIHRvIGNvcHkuIFRyYWRlIHNpZ25hbHMgd2l0aCBhIGxhcmdlciB2b2x1bWUgd2lsbCBiZSBjb3BpZWRcbiAgICogd2l0aCBtYXhpbXVtIHZvbHVtZSBpbnN0ZWFkXG4gICAqL1xuXG4gIC8qKlxuICAgKiBQb3J0Zm9saW8gc3RyYXRlZ3ksIGkuZS4gdGhlIHN0cmF0ZWd5IHdoaWNoIGluY2x1ZGVzIGEgc2V0IG9mIG90aGVyIHN0cmF0ZWdpZXNcbiAgICogQHR5cGVkZWYge0NvcHlGYWN0b3J5UG9ydGZvbGlvU3RyYXRlZ3lVcGRhdGV9IENvcHlGYWN0b3J5UG9ydGZvbGlvU3RyYXRlZ3lcbiAgICogQHByb3BlcnR5IHtTdHJpbmd9IF9pZCB1bmlxdWUgc3RyYXRlZ3kgaWRcbiAgICogQHByb3BlcnR5IHtOdW1iZXJ9IHBsYXRmb3JtQ29tbWlzc2lvblJhdGUgY29tbWlzc2lvbiByYXRlIHRoZSBwbGF0Zm9ybSBjaGFyZ2VzIGZvciBzdHJhdGVneSBjb3B5aW5nLCBhcHBsaWVkIHRvXG4gICAqIGNvbW1pc3Npb25zIGNoYXJnZWQgYnkgcHJvdmlkZXIuIFRoaXMgY29tbWlzc2lvbiBhcHBsaWVzIG9ubHkgdG8gYWNjb3VudHMgbm90IG1hbmFnZWQgZGlyZWN0bHkgYnkgcHJvdmlkZXIuIFNob3VsZFxuICAgKiBiZSBmcmFjdGlvbiBvZiAxXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBbY2xvc2VPblJlbW92YWxNb2RlXSBwb3NpdGlvbiBjbG9zZSBtb2RlIG9uIHN0cmF0ZWd5IG9yIHN1YnNjcmlwdGlvbiByZW1vdmFsLiBQcmVzZXJ2ZSBtZWFuc1xuICAgKiB0aGF0IHBvc2l0aW9ucyB3aWxsIG5vdCBiZSBjbG9zZWQgYW5kIHdpbGwgbm90IGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnkuIGNsb3NlLWdyYWNlZnVsbHktYnktcG9zaXRpb24gbWVhbnNcbiAgICogdGhhdCBwb3NpdGlvbnMgd2lsbCBjb250aW51ZSB0byBiZSBtYW5hZ2VkIGJ5IENvcHlGYWN0b3J5LCBidXQgb25seSBjbG9zZSBzaWduYWxzIHdpbGwgYmUgY29waWVkLlxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCBtZWFucyB0aGF0IHBvc2l0aW9ucyB3aWxsIGNvbnRpbnVlIHRvIGJlIG1hbmFnZWQgYnkgQ29weUZhY3RvcnksIGJ1dCBvbmx5IGNsb3NlXG4gICAqIHNpZ25hbHMgd2lsbCBiZSBjb3BpZWQgb3Igc2lnbmFscyB0byBvcGVuIHBvc2l0aW9ucyBmb3IgdGhlIHN5bWJvbHMgd2hpY2ggYWxyZWFkeSBoYXZlIHBvc2l0aW9ucyBvcGVuZWQuXG4gICAqIGNsb3NlLWltbWVkaWF0ZWx5IG1lYW5zIHRoYXQgYWxsIHBvc2l0aW9ucyB3aWxsIGJlIGNsb3NlZCBpbW1lZGlhdGVseS4gRGVmYXVsdCBpcyBjbG9zZS1pbW1lZGlhdGVseS5cbiAgICogVGhpcyBmaWVsZCBjYW4gYmUgY2hhbmdlZCB2aWEgcmVtb3ZlIHBvdGZvbGlvIG1lbWJlciBBUEkgb25seSwgb25lIG9mIHByZXNlcnZlLCBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXBvc2l0aW9uLFxuICAgKiBjbG9zZS1ncmFjZWZ1bGx5LWJ5LXN5bWJvbCwgY2xvc2UtaW1tZWRpYXRlbHlcbiAgICovXG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBDb3B5RmFjdG9yeSBjb3B5IHBvcnRmb2xpbyBzdHJhdGVnaWVzLiBTZWVcbiAgICogaHR0cHM6Ly9tZXRhYXBpLmNsb3VkL2RvY3MvY29weWZhY3RvcnkvcmVzdEFwaS9hcGkvY29uZmlndXJhdGlvbi9nZXRQb3J0Zm9saW9TdHJhdGVnaWVzL1xuICAgKiBAcGFyYW0ge0Jvb2xlYW59IFtpbmNsdWRlUmVtb3ZlZF0gZmxhZyBpbnN0cnVjdGluZyB0byBpbmNsdWRlIHJlbW92ZWQgcG9ydGZvbGlvIHN0cmF0ZWdpZXMgaW4gcmVzdWx0c1xuICAgKiBAcGFyYW0ge051bWJlcn0gW2xpbWl0XSBwYWdpbmF0aW9uIGxpbWl0XG4gICAqIEBwYXJhbSB7TnVtYmVyfSBbb2Zmc2V0XSBjb3B5IHRyYWRpbmcgc3RyYXRlZ3kgaWRcbiAgICogQHJldHVybiB7UHJvbWlzZTxBcnJheTxDb3B5RmFjdG9yeVBvcnRmb2xpb1N0cmF0ZWd5Pj59IHByb21pc2UgcmVzb2x2aW5nIHdpdGggQ29weUZhY3RvcnkgcG9ydGZvbGlvIHN0cmF0ZWdpZXNcbiAgICogZm91bmRcbiAgICovXG4gIGdldFBvcnRmb2xpb1N0cmF0ZWdpZXMoaW5jbHVkZVJlbW92ZWQsIGxpbWl0LCBvZmZzZXQpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcignZ2V0UG9ydGZvbGlvU3RyYXRlZ2llcycpO1xuICAgIH1cbiAgICBsZXQgcXMgPSB7fTtcbiAgICBpZihpbmNsdWRlUmVtb3ZlZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5pbmNsdWRlUmVtb3ZlZCA9IGluY2x1ZGVSZW1vdmVkO1xuICAgIH1cbiAgICBpZihsaW1pdCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5saW1pdCA9IGxpbWl0O1xuICAgIH1cbiAgICBpZihvZmZzZXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcXMub2Zmc2V0ID0gb2Zmc2V0O1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vcG9ydGZvbGlvLXN0cmF0ZWdpZXNgLFxuICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIHFzLFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgQ29weUZhY3RvcnkgY29weSBwb3J0Zm9saW8gc3RyYXRlZ3kgYnkgaWQuIFNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jb3B5ZmFjdG9yeS9yZXN0QXBpL2FwaS9jb25maWd1cmF0aW9uL2dldFBvcnRmb2xpb1N0cmF0ZWd5L1xuICAgKiBAcGFyYW0ge3N0cmluZ30gcG9ydGZvbGlvSWQgcG9ydGZvbGlvIHN0cmF0ZWd5IGlkXG4gICAqIEByZXR1cm4ge1Byb21pc2U8Q29weUZhY3RvcnlQb3J0Zm9saW9TdHJhdGVneT59IHByb21pc2UgcmVzb2x2aW5nIHdpdGggQ29weUZhY3RvcnkgcG9ydGZvbGlvIHN0cmF0ZWd5IGZvdW5kXG4gICAqL1xuICBnZXRQb3J0Zm9saW9TdHJhdGVneShwb3J0Zm9saW9JZCkge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdnZXRQb3J0Zm9saW9TdHJhdGVneScpO1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vcG9ydGZvbGlvLXN0cmF0ZWdpZXMvJHtwb3J0Zm9saW9JZH1gLFxuICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIENvcHlGYWN0b3J5IHBvcnRmb2xpbyBzdHJhdGVneS4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vdXBkYXRlUG9ydGZvbGlvU3RyYXRlZ3kvXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBwb3J0Zm9saW9JZCBjb3B5IHRyYWRpbmcgcG9ydGZvbGlvIHN0cmF0ZWd5IGlkXG4gICAqIEBwYXJhbSB7Q29weUZhY3RvcnlQb3J0Zm9saW9TdHJhdGVneVVwZGF0ZX0gcG9ydGZvbGlvIHBvcnRmb2xpbyBzdHJhdGVneSB1cGRhdGVcbiAgICogQHJldHVybiB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBwb3J0Zm9saW8gc3RyYXRlZ3kgaXMgdXBkYXRlZFxuICAgKi9cbiAgdXBkYXRlUG9ydGZvbGlvU3RyYXRlZ3kocG9ydGZvbGlvSWQsIHBvcnRmb2xpbykge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCd1cGRhdGVQb3J0Zm9saW9TdHJhdGVneScpO1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vcG9ydGZvbGlvLXN0cmF0ZWdpZXMvJHtwb3J0Zm9saW9JZH1gLFxuICAgICAgbWV0aG9kOiAnUFVUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIGJvZHk6IHBvcnRmb2xpbyxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIENvcHlGYWN0b3J5IHBvcnRmb2xpbyBzdHJhdGVneS4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vcmVtb3ZlUG9ydGZvbGlvU3RyYXRlZ3kvXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBwb3J0Zm9saW9JZCBwb3J0Zm9saW8gc3RyYXRlZ3kgaWRcbiAgICogQHBhcmFtIHtDb3B5RmFjdG9yeUNsb3NlSW5zdHJ1Y3Rpb25zfSBbY2xvc2VJbnN0cnVjdGlvbnNdIHN0cmF0ZWd5IGNsb3NlIGluc3RydWN0aW9uc1xuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIHBvcnRmb2xpbyBzdHJhdGVneSBpcyByZW1vdmVkXG4gICAqL1xuICByZW1vdmVQb3J0Zm9saW9TdHJhdGVneShwb3J0Zm9saW9JZCwgY2xvc2VJbnN0cnVjdGlvbnMpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcigncmVtb3ZlUG9ydGZvbGlvU3RyYXRlZ3knKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIHVybDogYCR7dGhpcy5faG9zdH0vdXNlcnMvY3VycmVudC9jb25maWd1cmF0aW9uL3BvcnRmb2xpby1zdHJhdGVnaWVzLyR7cG9ydGZvbGlvSWR9YCxcbiAgICAgIG1ldGhvZDogJ0RFTEVURScsXG4gICAgICBoZWFkZXJzOiB7XG4gICAgICAgICdhdXRoLXRva2VuJzogdGhpcy5fdG9rZW5cbiAgICAgIH0sXG4gICAgICBib2R5OiBjbG9zZUluc3RydWN0aW9ucyxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIENvcHlGYWN0b3J5IHBvcnRmb2xpbyBzdHJhdGVneSBtZW1iZXIuIFNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jb3B5ZmFjdG9yeS9yZXN0QXBpL2FwaS9jb25maWd1cmF0aW9uL3JlbW92ZVBvcnRmb2xpb1N0cmF0ZWd5TWVtYmVyL1xuICAgKiBAcGFyYW0ge1N0cmluZ30gcG9ydGZvbGlvSWQgcG9ydGZvbGlvIHN0cmF0ZWd5IGlkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBzdHJhdGVneUlkIGlkIG9mIHRoZSBzdHJhdGVneSB0byBkZWxldGUgbWVtYmVyIGZvclxuICAgKiBAcGFyYW0ge0NvcHlGYWN0b3J5Q2xvc2VJbnN0cnVjdGlvbnN9IFtjbG9zZUluc3RydWN0aW9uc10gc3RyYXRlZ3kgY2xvc2UgaW5zdHJ1Y3Rpb25zXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2UgcmVzb2x2aW5nIHdoZW4gcG9ydGZvbGlvIHN0cmF0ZWd5IGlzIHJlbW92ZWRcbiAgICovXG4gIHJlbW92ZVBvcnRmb2xpb1N0cmF0ZWd5TWVtYmVyKHBvcnRmb2xpb0lkLCBzdHJhdGVneUlkLCBjbG9zZUluc3RydWN0aW9ucykge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdyZW1vdmVQb3J0Zm9saW9TdHJhdGVneU1lbWJlcicpO1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vcG9ydGZvbGlvLXN0cmF0ZWdpZXMvJHtwb3J0Zm9saW9JZH0vbWVtYmVycy8ke3N0cmF0ZWd5SWR9YCxcbiAgICAgIG1ldGhvZDogJ0RFTEVURScsXG4gICAgICBoZWFkZXJzOiB7XG4gICAgICAgICdhdXRoLXRva2VuJzogdGhpcy5fdG9rZW5cbiAgICAgIH0sXG4gICAgICBib2R5OiBjbG9zZUluc3RydWN0aW9ucyxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBDb3B5RmFjdG9yeSBzdWJzY3JpYmVycyB0aGUgdXNlciBoYXMgY29uZmlndXJlZC4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2hpc3RvcnkvZ2V0U3Vic2NyaWJlcnMvXG4gICAqIEBwYXJhbSB7Qm9vbGVhbn0gW2luY2x1ZGVSZW1vdmVkXSBmbGFnIGluc3RydWN0aW5nIHRvIGluY2x1ZGUgcmVtb3ZlZCBzdWJzY3JpYmVycyBpbiByZXN1bHRzXG4gICAqIEBwYXJhbSB7TnVtYmVyfSBbbGltaXRdIHBhZ2luYXRpb24gbGltaXRcbiAgICogQHBhcmFtIHtOdW1iZXJ9IFtvZmZzZXRdIGNvcHkgdHJhZGluZyBzdHJhdGVneSBpZFxuICAgKiBAcmV0dXJuIHtQcm9taXNlPEFycmF5PENvcHlGYWN0b3J5U3Vic2NyaWJlcj4+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIHN1YnNjcmliZXJzIGZvdW5kXG4gICAqL1xuICBnZXRTdWJzY3JpYmVycyhpbmNsdWRlUmVtb3ZlZCwgbGltaXQsIG9mZnNldCkge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdnZXRTdWJzY3JpYmVycycpO1xuICAgIH1cbiAgICBsZXQgcXMgPSB7fTtcbiAgICBpZihpbmNsdWRlUmVtb3ZlZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5pbmNsdWRlUmVtb3ZlZCA9IGluY2x1ZGVSZW1vdmVkO1xuICAgIH1cbiAgICBpZihsaW1pdCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBxcy5saW1pdCA9IGxpbWl0O1xuICAgIH1cbiAgICBpZihvZmZzZXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcXMub2Zmc2V0ID0gb2Zmc2V0O1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vc3Vic2NyaWJlcnNgLFxuICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIHFzLFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIENvcHlGYWN0b3J5IHN1YnNjcmliZXIgYnkgaWQuIFNlZVxuICAgKiBodHRwczovL21ldGFhcGkuY2xvdWQvZG9jcy9jb3B5ZmFjdG9yeS9yZXN0QXBpL2FwaS9jb25maWd1cmF0aW9uL2dldFN1YnNjcmliZXIvXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBzdWJzY3JpYmVySWQgc3Vic2NyaWJlciBpZFxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTxDb3B5RmFjdG9yeVN1YnNjcmliZXI+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIHN1YnNjcmliZXIgZm91bmRcbiAgICovXG4gIGdldFN1YnNjcmliZXIoc3Vic2NyaWJlcklkKSB7XG4gICAgaWYgKHRoaXMuX2lzTm90Snd0VG9rZW4oKSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2hhbmRsZU5vQWNjZXNzRXJyb3IoJ2dldFN1YnNjcmliZXInKTtcbiAgICB9XG4gICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgIHVybDogYCR7dGhpcy5faG9zdH0vdXNlcnMvY3VycmVudC9jb25maWd1cmF0aW9uL3N1YnNjcmliZXJzLyR7c3Vic2NyaWJlcklkfWAsXG4gICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgaGVhZGVyczoge1xuICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICB9LFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIENvcHlGYWN0b3J5IHN1YnNjcmliZXIgY29uZmlndXJhdGlvbi4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vdXBkYXRlU3Vic2NyaWJlci9cbiAgICogQHBhcmFtIHtTdHJpbmd9IHN1YnNjcmliZXJJZCBzdWJzY3JpYmVyIGlkXG4gICAqIEBwYXJhbSB7Q29weUZhY3RvcnlTdWJzY3JpYmVyVXBkYXRlfSBzdWJzY3JpYmVyIHN1YnNjcmliZXIgdXBkYXRlXG4gICAqIEByZXR1cm5zIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIHN1YnNjcmliZXIgaXMgdXBkYXRlZFxuICAgKi9cbiAgdXBkYXRlU3Vic2NyaWJlcihzdWJzY3JpYmVySWQsIHN1YnNjcmliZXIpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcigndXBkYXRlU3Vic2NyaWJlcicpO1xuICAgIH1cbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgdXJsOiBgJHt0aGlzLl9ob3N0fS91c2Vycy9jdXJyZW50L2NvbmZpZ3VyYXRpb24vc3Vic2NyaWJlcnMvJHtzdWJzY3JpYmVySWR9YCxcbiAgICAgIG1ldGhvZDogJ1BVVCcsXG4gICAgICBoZWFkZXJzOiB7XG4gICAgICAgICdhdXRoLXRva2VuJzogdGhpcy5fdG9rZW5cbiAgICAgIH0sXG4gICAgICBib2R5OiBzdWJzY3JpYmVyLFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHN1YnNjcmliZXIgY29uZmlndXJhdGlvbi4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vcmVtb3ZlU3Vic2NyaWJlci9cbiAgICogQHBhcmFtIHtTdHJpbmd9IHN1YnNjcmliZXJJZCBzdWJzY3JpYmVyIGlkXG4gICAqIEBwYXJhbSB7Q29weUZhY3RvcnlDbG9zZUluc3RydWN0aW9uc30gW2Nsb3NlSW5zdHJ1Y3Rpb25zXSBzdWJzY3JpYmVyIGNsb3NlIGluc3RydWN0aW9uc1xuICAgKiBAcmV0dXJucyB7UHJvbWlzZX0gcHJvbWlzZSByZXNvbHZpbmcgd2hlbiBzdWJzY3JpYmVyIGlzIHJlbW92ZWRcbiAgICovXG4gIHJlbW92ZVN1YnNjcmliZXIoc3Vic2NyaWJlcklkLCBjbG9zZUluc3RydWN0aW9ucykge1xuICAgIGlmICh0aGlzLl9pc05vdEp3dFRva2VuKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLl9oYW5kbGVOb0FjY2Vzc0Vycm9yKCdyZW1vdmVTdWJzY3JpYmVyJyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvY29uZmlndXJhdGlvbi9zdWJzY3JpYmVycy8ke3N1YnNjcmliZXJJZH1gLFxuICAgICAgbWV0aG9kOiAnREVMRVRFJyxcbiAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgJ2F1dGgtdG9rZW4nOiB0aGlzLl90b2tlblxuICAgICAgfSxcbiAgICAgIGJvZHk6IGNsb3NlSW5zdHJ1Y3Rpb25zLFxuICAgICAganNvbjogdHJ1ZVxuICAgIH07XG4gICAgcmV0dXJuIHRoaXMuX2h0dHBDbGllbnQucmVxdWVzdChvcHRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3Vic2NyaXB0aW9uIG9mIHN1YnNjcmliZXIgdG8gYSBzdHJhdGVneS4gU2VlXG4gICAqIGh0dHBzOi8vbWV0YWFwaS5jbG91ZC9kb2NzL2NvcHlmYWN0b3J5L3Jlc3RBcGkvYXBpL2NvbmZpZ3VyYXRpb24vcmVtb3ZlU3Vic2NyaXB0aW9uL1xuICAgKiBAcGFyYW0ge1N0cmluZ30gc3Vic2NyaWJlcklkIHN1YnNjcmliZXIgaWRcbiAgICogQHBhcmFtIHtTdHJpbmd9IHN0cmF0ZWd5SWQgc3RyYXRlZ3kgaWRcbiAgICogQHBhcmFtIHtDb3B5RmFjdG9yeUNsb3NlSW5zdHJ1Y3Rpb25zfSBbY2xvc2VJbnN0cnVjdGlvbnNdIHN1YnNjcmliZXIgY2xvc2UgaW5zdHJ1Y3Rpb25zXG4gICAqIEByZXR1cm5zIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIHN1YnNjcmliZXIgaXMgcmVtb3ZlZFxuICAgKi9cbiAgcmVtb3ZlU3Vic2NyaXB0aW9uKHN1YnNjcmliZXJJZCwgc3RyYXRlZ3lJZCwgY2xvc2VJbnN0cnVjdGlvbnMpIHtcbiAgICBpZiAodGhpcy5faXNOb3RKd3RUb2tlbigpKSB7XG4gICAgICByZXR1cm4gdGhpcy5faGFuZGxlTm9BY2Nlc3NFcnJvcigncmVtb3ZlU3Vic2NyaXB0aW9uJyk7XG4gICAgfVxuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICB1cmw6IGAke3RoaXMuX2hvc3R9L3VzZXJzL2N1cnJlbnQvY29uZmlndXJhdGlvbi9zdWJzY3JpYmVycy8ke3N1YnNjcmliZXJJZH0vc3Vic2NyaXB0aW9ucy8ke3N0cmF0ZWd5SWR9YCxcbiAgICAgIG1ldGhvZDogJ0RFTEVURScsXG4gICAgICBoZWFkZXJzOiB7XG4gICAgICAgICdhdXRoLXRva2VuJzogdGhpcy5fdG9rZW5cbiAgICAgIH0sXG4gICAgICBib2R5OiBjbG9zZUluc3RydWN0aW9ucyxcbiAgICAgIGpzb246IHRydWVcbiAgICB9O1xuICAgIHJldHVybiB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cyk7XG4gIH1cblxufVxuIl19