'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HttpClientMock = undefined;

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _requestPromiseAny = require('request-promise-any');

var _requestPromiseAny2 = _interopRequireDefault(_requestPromiseAny);

var _errorHandler = require('./errorHandler');

var _timeoutError = require('./timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * HTTP client library based on request-promise
 */
class HttpClient {

  /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   */

  /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */
  constructor(timeout = 60, retryOpts = {}) {
    this._timeout = timeout * 1000;
    this._retries = retryOpts.retries || 5;
    this._minRetryDelay = (retryOpts.minDelayInSeconds || 1) * 1000;
    this._maxRetryDelay = (retryOpts.maxDelayInSeconds || 30) * 1000;
  }

  /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */
  async request(options, retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries) {
    options.timeout = this._timeout;
    let retryAfterSeconds = 0;
    options.callback = (e, res) => {
      if (res && res.statusCode === 202) {
        retryAfterSeconds = res.headers['retry-after'];
      }
    };
    let body;
    try {
      body = await this._makeRequest(options);
    } catch (err) {
      retryCounter = await this._handleError(err, retryCounter, endTime);
      return this.request(options, retryCounter, endTime);
    }
    if (retryAfterSeconds) {
      await this._handleRetry(endTime, retryAfterSeconds * 1000);
      body = await this.request(options, retryCounter, endTime);
    }
    return body;
  }

  _makeRequest(options) {
    return (0, _requestPromiseAny2.default)(options);
  }

  async _wait(pause) {
    await new _promise2.default(res => setTimeout(res, pause));
  }

  async _handleRetry(endTime, retryAfter) {
    if (endTime > Date.now() + retryAfter) {
      await this._wait(retryAfter);
    } else {
      throw new _timeoutError2.default('Timed out waiting for the end of the process of calculating metrics');
    }
  }

  async _handleError(err, retryCounter, endTime) {
    const error = this._convertError(err);
    if (['ConflictError', 'InternalError', 'ApiError', 'TimeoutError'].includes(error.name) && retryCounter < this._retries) {
      const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
      await this._wait(pause);
      return retryCounter + 1;
    } else if (error.name === 'TooManyRequestsError') {
      const retryTime = Date.parse(error.metadata.recommendedRetryTime);
      if (retryTime < endTime) {
        await this._wait(retryTime - Date.now());
        return retryCounter;
      }
    }
    throw error;
  }

  // eslint-disable-next-line complexity
  _convertError(err) {
    err.error = err.error || {};
    let status = err.statusCode || err.status;
    switch (status) {
      case 400:
        return new _errorHandler.ValidationError(err.error.message || err.message, err.error.details || err.details);
      case 401:
        return new _errorHandler.UnauthorizedError(err.error.message || err.message);
      case 403:
        return new _errorHandler.ForbiddenError(err.error.message || err.message);
      case 404:
        return new _errorHandler.NotFoundError(err.error.message || err.message);
      case 409:
        return new _errorHandler.ConflictError(err.error.message || err.message);
      case 429:
        return new _errorHandler.TooManyRequestsError(err.error.message || err.message, err.error.metadata || err.metadata);
      case 500:
        return new _errorHandler.InternalError(err.error.message || err.message);
      default:
        return new _errorHandler.ApiError(_errorHandler.ApiError, err.error.message || err.message, status);
    }
  }

}

exports.default = HttpClient; /**
                               * HTTP client service mock for tests
                               */

class HttpClientMock extends HttpClient {

  /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */
  constructor(requestFn, timeout, retryOpts) {
    super(timeout, retryOpts);
    this._requestFn = requestFn;
  }

  _makeRequest() {
    return this._requestFn.apply(this, arguments);
  }

}
exports.HttpClientMock = HttpClientMock;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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