'use strict';

/**
 * Defines interface for a synchronization listener class
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class SynchronizationListener {

  /**
   * Returns instance number of instance index
   * @param {String} instanceIndex instance index
   */
  getInstanceNumber(instanceIndex) {
    return typeof instanceIndex === 'string' ? Number(instanceIndex.split(':')[0]) : undefined;
  }

  /**
   * Returns host name of instance index
   * @param {String} instanceIndex instance index
   */
  getHostName(instanceIndex) {
    return typeof instanceIndex === 'string' ? instanceIndex.split(':')[1] : undefined;
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onConnected(instanceIndex, replicas) {}

  /**
   * Server-side application health status
   * @typedef {Object} healthStatus
   * @property {boolean} [restApiHealthy] flag indicating that REST API is healthy
   * @property {boolean} [copyFactorySubscriberHealthy] flag indicating that CopyFactory subscriber is healthy
   * @property {boolean} [copyFactoryProviderHealthy] flag indicating that CopyFactory provider is healthy
   */

  /**
   * Invoked when a server-side application health status is received from MetaApi
   * @param {String} instanceIndex index of an account instance connected
   * @param {HealthStatus} status server-side application health status
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHealthStatus(instanceIndex, status) {}

  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDisconnected(instanceIndex) {}

  /**
   * Invoked when broker connection satus have changed
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} connected is MetaTrader terminal is connected to broker
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onBrokerConnectionStatusChanged(instanceIndex, connected) {}

  /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} specificationsUpdated whether specifications are going to be updated during synchronization
   * @param {Boolean} positionsUpdated whether positions are going to be updated during synchronization
   * @param {Boolean} ordersUpdated whether orders are going to be updated during synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSynchronizationStarted(instanceIndex, specificationsUpdated, positionsUpdated, ordersUpdated) {}

  /**
   * Invoked when MetaTrader account information is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderAccountInformation} accountInformation updated MetaTrader account information
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onAccountInformationUpdated(instanceIndex, accountInformation) {}

  /**
   * Invoked when the positions are replaced as a result of initial terminal state synchronization. This method
   * will be invoked only if server thinks the data was updated, otherwise invocation can be skipped
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderPosition>} positions updated array of positions
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionsReplaced(instanceIndex, positions) {}

  /**
   * Invoked when position synchronization fnished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionsSynchronized(instanceIndex, synchronizationId) {}

  /**
   * Invoked when MetaTrader position is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderPosition} position updated MetaTrader position
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionUpdated(instanceIndex, position) {}

  /**
   * Invoked when MetaTrader position is removed
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} positionId removed MetaTrader position id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionRemoved(instanceIndex, positionId) {}

  /**
   * Invoked when the pending orders are replaced as a result of initial terminal state synchronization. This method
   * will be invoked only if server thinks the data was updated, otherwise invocation can be skipped
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderOrder>} orders updated array of pending orders
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrdersReplaced(instanceIndex, orders) {}

  /**
   * Invoked when MetaTrader pending order is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} order updated MetaTrader pending order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrderUpdated(instanceIndex, order) {}

  /**
   * Invoked when MetaTrader pending order is completed (executed or canceled)
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} orderId completed MetaTrader pending order id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrderCompleted(instanceIndex, orderId) {}

  /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrdersSynchronized(instanceIndex, synchronizationId) {}

  /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrderAdded(instanceIndex, historyOrder) {}

  /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {}

  /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealAdded(instanceIndex, deal) {}

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized(instanceIndex, synchronizationId) {}

  /**
   * Invoked when a symbol specification was updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderSymbolSpecification} specification updated MetaTrader symbol specification
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolSpecificationUpdated(instanceIndex, specification) {}

  /**
   * Invoked when a symbol specification was removed
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} symbol removed symbol
   * @returns {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolSpecificationRemoved(instanceIndex, symbol) {}

  /**
   * Invoked when a symbol specifications were updated
   * @param {String} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<String>} removedSymbols removed symbols
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolSpecificationsUpdated(instanceIndex, specifications, removedSymbols) {}

  /**
   * Invoked when a symbol price was updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderSymbolPrice} price updated MetaTrader symbol price
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolPriceUpdated(instanceIndex, price) {}

  /**
   * Invoked when prices for several symbols were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderSymbolPrice>} prices updated MetaTrader symbol prices
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   * @param {Number} accountCurrencyExchangeRate current exchange rate of account currency into USD
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolPricesUpdated(instanceIndex, prices, equity, margin, freeMargin, marginLevel, accountCurrencyExchangeRate) {}

  /**
   * Invoked when symbol candles were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderCandle>} candles updated MetaTrader symbol candles
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   * @param {Number} accountCurrencyExchangeRate current exchange rate of account currency into USD
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onCandlesUpdated(instanceIndex, candles, equity, margin, freeMargin, marginLevel, accountCurrencyExchangeRate) {}

  /**
   * Invoked when symbol ticks were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderTick>} ticks updated MetaTrader symbol ticks
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   * @param {Number} accountCurrencyExchangeRate current exchange rate of account currency into USD
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onTicksUpdated(instanceIndex, ticks, equity, margin, freeMargin, marginLevel, accountCurrencyExchangeRate) {}

  /**
   * Invoked when order books were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderBook>} books updated MetaTrader order books
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   * @param {Number} accountCurrencyExchangeRate current exchange rate of account currency into USD
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onBooksUpdated(instanceIndex, books, equity, margin, freeMargin, marginLevel, accountCurrencyExchangeRate) {}

  /**
   * Invoked when subscription downgrade has occurred
   * @param {String} instanceIndex index of an account instance connected
   * @param {string} symbol symbol to update subscriptions for
   * @param {Array<MarketDataSubscription>} updates array of market data subscription to update
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSubscriptionDowngraded(instanceIndex, symbol, updates, unsubscriptions) {}

  /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   */
  async onStreamClosed(instanceIndex) {}

}
exports.default = SynchronizationListener;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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