'use strict';

var _create = require('babel-runtime/core-js/object/create');

var _create2 = _interopRequireDefault(_create);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _connectionHealthMonitor = require('./connectionHealthMonitor');

var _connectionHealthMonitor2 = _interopRequireDefault(_connectionHealthMonitor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {ConnectionHealthMonitor}
 */
describe('ConnectionHealthMonitor', () => {
  let healthMonitor, sandbox, prices, connection;
  let brokerTimes = ['2020-10-05 09:00:00.000', '2020-10-10 10:00:00.000'];
  let clock, updateInterval;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
    const mockMath = (0, _create2.default)(global.Math);
    mockMath.random = () => 0.4833333;
    global.Math = mockMath;
  });

  beforeEach(() => {
    clock = _sinon2.default.useFakeTimers({
      now: new Date('2020-10-05 10:00:00.000')
    });
    connection = {
      account: { id: 'id' },
      subscribedSymbols: ['EURUSD'],
      terminalState: {
        specification: () => {},
        connected: true,
        connectedToBroker: true
      },
      synchronized: true
    };
    sandbox.stub(connection.terminalState, 'specification').returns({ quoteSessions: { 'MONDAY': [{ from: '08:00:00.000', to: '17:00:00.000' }] } });
    healthMonitor = new _connectionHealthMonitor2.default(connection);
    prices = [{
      symbol: 'EURUSD',
      brokerTime: brokerTimes[0]
    }, {
      symbol: 'EURUSD',
      brokerTime: brokerTimes[1]
    }];
    updateInterval = setInterval(() => healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]), 1000);
  });

  afterEach(() => {
    clock.restore();
    sandbox.restore();
  });

  /**
   * @test {ConnectionHealthMonitor#uptime}
   */
  describe('uptime', () => {

    /**
     * @test {ConnectionHealthMonitor#uptime}
     */
    it('should return 100 uptime', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      await clock.tickAsync(60000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 100, '1d': 100, '1w': 100 });
    });

    /**
     * @test {ConnectionHealthMonitor#uptime}
     */
    it('should return average uptime', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      await clock.tickAsync(31000);
      clearInterval(updateInterval);
      await clock.tickAsync(120000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 60, '1d': 60, '1w': 60 });
    });

    /**
     * @test {ConnectionHealthMonitor#uptime}
     */
    it('should check connection for downtime', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      await clock.tickAsync(120000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 100, '1d': 100, '1w': 100 });
      connection.terminalState.connected = false;
      await clock.tickAsync(120000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 50, '1d': 50, '1w': 50 });
      connection.terminalState.connected = true;
      connection.terminalState.connectedToBroker = false;
      await clock.tickAsync(240000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 25, '1d': 25, '1w': 25 });
      connection.terminalState.connectedToBroker = true;
      connection.synchronized = false;
      await clock.tickAsync(120000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 20, '1d': 20, '1w': 20 });
      connection.synchronized = true;
      await clock.tickAsync(360000);
      _sinon2.default.assert.match(healthMonitor.uptime, { '1h': 50, '1d': 50, '1w': 50 });
    });
  });

  /**
   * @test {ConnectionHealthMonitor#healthStatus}
   */
  describe('healthStatus', () => {

    beforeEach(() => {
      healthMonitor._quotesHealthy = true;
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should return ok status', async () => {
      _sinon2.default.assert.match(healthMonitor.healthStatus, {
        connected: true,
        connectedToBroker: true,
        healthy: true,
        message: 'Connection to broker is stable. No health issues detected.',
        quoteStreamingHealthy: true,
        synchronized: true
      });
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should return error status with one message', async () => {
      connection.terminalState.connectedToBroker = false;
      _sinon2.default.assert.match(healthMonitor.healthStatus, {
        connected: true,
        connectedToBroker: false,
        healthy: false,
        message: 'Connection is not healthy because connection to broker is not established or lost.',
        quoteStreamingHealthy: true,
        synchronized: true
      });
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should return error status with multiple messages', async () => {
      connection.terminalState.connected = false;
      connection.terminalState.connectedToBroker = false;
      connection.synchronized = false;
      _sinon2.default.assert.match(healthMonitor.healthStatus, {
        connected: false,
        connectedToBroker: false,
        healthy: false,
        message: 'Connection is not healthy because connection to API server is not established or lost and ' + 'connection to broker is not established or lost ' + 'and local terminal state is not synchronized to broker.',
        quoteStreamingHealthy: true,
        synchronized: false
      });
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should show as healthy if recently updated and in session', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      await clock.tickAsync(91000);
      _sinon2.default.assert.match(healthMonitor.healthStatus.quoteStreamingHealthy, true);
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should show as not healthy if old update and in session', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      clearInterval(updateInterval);
      await clock.tickAsync(91000);
      _sinon2.default.assert.match(healthMonitor.healthStatus.quoteStreamingHealthy, false);
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should show as healthy if not in session', async () => {
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[1]);
      clearInterval(updateInterval);
      await clock.tickAsync(91000);
      _sinon2.default.assert.match(healthMonitor.healthStatus.quoteStreamingHealthy, true);
    });

    /**
     * @test {ConnectionHealthMonitor#healthStatus}
     */
    it('should show as healthy if no symbols', async () => {
      healthMonitor._connection.subscribedSymbols = [];
      healthMonitor.onSymbolPriceUpdated('1:ps-mpa-1', prices[0]);
      clearInterval(updateInterval);
      await clock.tickAsync(91000);
      _sinon2.default.assert.match(healthMonitor.healthStatus.quoteStreamingHealthy, true);
    });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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