'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const HistoryFileManager = require('./nodeManager');

/**
 * Helper function to read saved history storage
 */
async function readHistoryStorageFile() {
  const storage = {
    deals: [],
    historyOrders: [],
    lastDealTimeByInstanceIndex: {},
    lastHistoryOrderTimeByInstanceIndex: {}
  };
  const isConfigExists = _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-config.bin');
  if (isConfigExists) {
    let config = JSON.parse(_fsExtra2.default.readFileSync('./.metaapi/accountId-application-config.bin', 'utf-8').toString('utf-8'));
    storage.lastDealTimeByInstanceIndex = config.lastDealTimeByInstanceIndex;
    storage.lastHistoryOrderTimeByInstanceIndex = config.lastHistoryOrderTimeByInstanceIndex;
  }
  const isDealsExists = _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-deals.bin');
  if (isDealsExists) {
    let deals = JSON.parse(_fsExtra2.default.readFileSync('./.metaapi/accountId-application-deals.bin', 'utf-8').toString('utf-8'));
    if (deals.length) {
      storage.deals = deals.map(deal => {
        deal.time = new Date(deal.time);
        return deal;
      });
    }
  }
  const isOrdersExists = _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-historyOrders.bin');
  if (isOrdersExists) {
    let historyOrders = JSON.parse(_fsExtra2.default.readFileSync('./.metaapi/accountId-application-historyOrders.bin', 'utf-8').toString('utf-8'));
    if (historyOrders.length) {
      storage.historyOrders = historyOrders.map(historyOrder => {
        historyOrder.time = new Date(historyOrder.time);
        historyOrder.doneTime = new Date(historyOrder.doneTime);
        return historyOrder;
      });
    }
  }
  return storage;
}

/**
 * @test {HistoryFileManager}
 */
describe('HistoryFileManager', () => {

  let storage;
  let fileManager;
  let testDeal;
  let testDeal2;
  let testDeal3;
  let testOrder;
  let testOrder2;
  let testOrder3;
  let testConfig;
  let sandbox;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
    _fsExtra2.default.ensureDir('./.metaapi');
  });

  beforeEach(async () => {
    storage = {};
    fileManager = new HistoryFileManager('accountId', 'application', storage);
    sandbox.stub(fileManager, 'startUpdateJob').returns();
    testDeal = { id: '37863643', type: 'DEAL_TYPE_BALANCE', magic: 0, time: new Date(100), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 1' };
    testDeal2 = { id: '37863644', type: 'DEAL_TYPE_SELL', magic: 1, time: new Date(200), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 2' };
    testDeal3 = { id: '37863645', type: 'DEAL_TYPE_BUY', magic: 2, time: new Date(300), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 3' };
    testOrder = { id: '61210463', type: 'ORDER_TYPE_SELL', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 0,
      time: new Date(50), doneTime: new Date(100), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206630', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    testOrder2 = { id: '61210464', type: 'ORDER_TYPE_BUY_LIMIT', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 1,
      time: new Date(75), doneTime: new Date(200), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206631', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    testOrder3 = { id: '61210465', type: 'ORDER_TYPE_BUY', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 1,
      time: new Date(100), doneTime: new Date(300), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206631', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    testConfig = {
      lastDealTimeByInstanceIndex: { '0': 1000000000000 },
      lastHistoryOrderTimeByInstanceIndex: { '0': 1000000000010 }
    };
  });

  after(() => {
    _fsExtra2.default.removeSync('./.metaapi');
  });

  afterEach(() => {
    sandbox.restore();
    _fsExtra2.default.removeSync('./.metaapi/accountId-application-deals.bin');
    _fsExtra2.default.removeSync('./.metaapi/accountId-application-historyOrders.bin');
  });

  /**
   * @test {HistoryFileManager#startUpdateJob}
   * @test {HistoryFileManager#stopUpdateJob}
   */
  it('should start and stop job', async () => {
    sandbox.restore();
    const clock = _sinon2.default.useFakeTimers();
    sandbox.stub(fileManager, 'updateDiskStorage').returns();
    fileManager.startUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledOnce(fileManager.updateDiskStorage);
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledTwice(fileManager.updateDiskStorage);
    fileManager.stopUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledTwice(fileManager.updateDiskStorage);
    fileManager.startUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledThrice(fileManager.updateDiskStorage);
    fileManager.stopUpdateJob();
    clock.restore();
  });

  /**
   * @test {HistoryFileManager#getHistoryFromDisk}
   */
  it('should read history from file', async () => {
    _fsExtra2.default.writeFileSync('./.metaapi/accountId-application-deals.bin', (0, _stringify2.default)([testDeal]), 'utf-8');
    _fsExtra2.default.writeFileSync('./.metaapi/accountId-application-historyOrders.bin', (0, _stringify2.default)([testOrder]), 'utf-8');
    _fsExtra2.default.writeFileSync('./.metaapi/accountId-application-config.bin', (0, _stringify2.default)(testConfig), 'utf-8');
    const history = await fileManager.getHistoryFromDisk();
    await new _promise2.default(res => setTimeout(res, 50));
    history.deals.should.match([testDeal]);
    history.historyOrders.should.match([testOrder]);
    history.lastDealTimeByInstanceIndex.should.match(testConfig.lastDealTimeByInstanceIndex);
    history.lastHistoryOrderTimeByInstanceIndex.should.match(testConfig.lastHistoryOrderTimeByInstanceIndex);
  });

  /**
   * @test {HistoryFileManager#updateDiskStorage}
   */
  it('should save items in a file', async () => {
    storage.deals = [testDeal];
    storage.historyOrders = [testOrder];
    storage.lastDealTimeByInstanceIndex = { '0': 1000000000000 };
    storage.lastHistoryOrderTimeByInstanceIndex = { '0': 1000000000010 };
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    fileManager.updateDiskStorage();
    await new _promise2.default(res => setTimeout(res, 50));
    const savedData = await readHistoryStorageFile();
    savedData.deals.should.match([testDeal]);
    savedData.historyOrders.should.match([testOrder]);
    savedData.lastDealTimeByInstanceIndex.should.match({ '0': 1000000000000 });
    savedData.lastHistoryOrderTimeByInstanceIndex.should.match({ '0': 1000000000010 });
  });

  /**
   * @test {HistoryFileManager#updateDiskStorage}
   */
  it('should replace Nth item in a file', async () => {
    storage.deals = [testDeal, testDeal2];
    storage.historyOrders = [testOrder, testOrder2];
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    fileManager.updateDiskStorage();
    await new _promise2.default(res => setTimeout(res, 50));
    testDeal2.magic = 100;
    testOrder2.magic = 100;
    fileManager.setStartNewOrderIndex(1);
    fileManager.setStartNewDealIndex(1);
    fileManager.updateDiskStorage();
    await new _promise2.default(res => setTimeout(res, 50));
    const savedData = await readHistoryStorageFile();
    savedData.deals.should.match([testDeal, testDeal2]);
    savedData.historyOrders.should.match([testOrder, testOrder2]);
  });

  /**
   * @test {HistoryFileManager#updateDiskStorage}
   */
  it('should replace all items in a file', async () => {
    storage.deals = [testDeal, testDeal2];
    storage.historyOrders = [testOrder, testOrder2];
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    await fileManager.updateDiskStorage();
    testDeal.magic = 100;
    testDeal2.magic = 100;
    testOrder.magic = 100;
    testOrder2.magic = 100;
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    await fileManager.updateDiskStorage();
    const savedData = await readHistoryStorageFile();
    savedData.deals.should.match([testDeal, testDeal2]);
    savedData.historyOrders.should.match([testOrder, testOrder2]);
  });

  /**
   * @test {HistoryFileManager#updateDiskStorage}
   */
  it('should append a new object to already saved ones', async () => {
    storage.deals = [testDeal, testDeal2];
    storage.historyOrders = [testOrder, testOrder2];
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    await fileManager.updateDiskStorage();
    storage.deals = [testDeal, testDeal2, testDeal3];
    storage.historyOrders = [testOrder, testOrder2, testOrder3];
    fileManager.setStartNewOrderIndex(2);
    fileManager.setStartNewDealIndex(2);
    await fileManager.updateDiskStorage();
    const savedData = await readHistoryStorageFile();
    savedData.deals.should.match([testDeal, testDeal2, testDeal3]);
    savedData.historyOrders.should.match([testOrder, testOrder2, testOrder3]);
  });

  /**
   * @test {HistoryFileManager#updateDiskStorage}
   */
  it('should not corrupt the disk storage if update called multiple times', async () => {
    storage.deals = [testDeal, testDeal2];
    storage.historyOrders = [testOrder, testOrder2];
    storage.lastDealTimeByInstanceIndex = { '0': 1000000000000 };
    storage.lastHistoryOrderTimeByInstanceIndex = { '0': 1000000000010 };
    fileManager.setStartNewOrderIndex(0);
    fileManager.setStartNewDealIndex(0);
    await fileManager.updateDiskStorage();
    storage.deals = [testDeal, testDeal2, testDeal3];
    storage.historyOrders = [testOrder, testOrder2, testOrder3];
    storage.lastDealTimeByInstanceIndex = { '1': 1000000000000 };
    storage.lastHistoryOrderTimeByInstanceIndex = { '1': 1000000000010 };
    fileManager.setStartNewOrderIndex(2);
    fileManager.setStartNewDealIndex(2);
    await _promise2.default.all([fileManager.updateDiskStorage(), fileManager.updateDiskStorage(), fileManager.updateDiskStorage(), fileManager.updateDiskStorage(), fileManager.updateDiskStorage()]);
    JSON.parse((await _fsExtra2.default.readFile('./.metaapi/accountId-application-historyOrders.bin')));
    JSON.parse((await _fsExtra2.default.readFile('./.metaapi/accountId-application-deals.bin')));
    JSON.parse((await _fsExtra2.default.readFile('./.metaapi/accountId-application-config.bin')));
  });

  /**
   * @test {HistoryFileManager#deleteStorageFromDisk}
   */
  it('should remove history from disk', async () => {
    await _fsExtra2.default.ensureFile('./.metaapi/accountId-application-historyOrders.bin');
    await _fsExtra2.default.ensureFile('./.metaapi/accountId-application-deals.bin');
    await _fsExtra2.default.ensureFile('./.metaapi/accountId-application-config.bin');
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-historyOrders.bin').should.equal(true);
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-deals.bin').should.equal(true);
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-config.bin').should.equal(true);
    await fileManager.deleteStorageFromDisk();
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-historyOrders.bin').should.equal(false);
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-deals.bin').should.equal(false);
    _fsExtra2.default.pathExistsSync('./.metaapi/accountId-application-config.bin').should.equal(false);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uL2xpYi9tZXRhQXBpL2hpc3RvcnlGaWxlTWFuYWdlci9ub2RlTWFuYWdlci5zcGVjLmVzNiJdLCJuYW1lcyI6WyJIaXN0b3J5RmlsZU1hbmFnZXIiLCJyZXF1aXJlIiwicmVhZEhpc3RvcnlTdG9yYWdlRmlsZSIsInN0b3JhZ2UiLCJkZWFscyIsImhpc3RvcnlPcmRlcnMiLCJsYXN0RGVhbFRpbWVCeUluc3RhbmNlSW5kZXgiLCJsYXN0SGlzdG9yeU9yZGVyVGltZUJ5SW5zdGFuY2VJbmRleCIsImlzQ29uZmlnRXhpc3RzIiwiZnMiLCJwYXRoRXhpc3RzU3luYyIsImNvbmZpZyIsIkpTT04iLCJwYXJzZSIsInJlYWRGaWxlU3luYyIsInRvU3RyaW5nIiwiaXNEZWFsc0V4aXN0cyIsImxlbmd0aCIsIm1hcCIsImRlYWwiLCJ0aW1lIiwiRGF0ZSIsImlzT3JkZXJzRXhpc3RzIiwiaGlzdG9yeU9yZGVyIiwiZG9uZVRpbWUiLCJkZXNjcmliZSIsImZpbGVNYW5hZ2VyIiwidGVzdERlYWwiLCJ0ZXN0RGVhbDIiLCJ0ZXN0RGVhbDMiLCJ0ZXN0T3JkZXIiLCJ0ZXN0T3JkZXIyIiwidGVzdE9yZGVyMyIsInRlc3RDb25maWciLCJzYW5kYm94IiwiYmVmb3JlIiwic2lub24iLCJjcmVhdGVTYW5kYm94IiwiZW5zdXJlRGlyIiwiYmVmb3JlRWFjaCIsInN0dWIiLCJyZXR1cm5zIiwiaWQiLCJ0eXBlIiwibWFnaWMiLCJjb21taXNzaW9uIiwic3dhcCIsInByb2ZpdCIsInBsYXRmb3JtIiwiY29tbWVudCIsInN0YXRlIiwic3ltYm9sIiwiY3VycmVudFByaWNlIiwidm9sdW1lIiwiY3VycmVudFZvbHVtZSIsInBvc2l0aW9uSWQiLCJhZnRlciIsInJlbW92ZVN5bmMiLCJhZnRlckVhY2giLCJyZXN0b3JlIiwiaXQiLCJjbG9jayIsInVzZUZha2VUaW1lcnMiLCJzdGFydFVwZGF0ZUpvYiIsInRpY2tBc3luYyIsImFzc2VydCIsImNhbGxlZE9uY2UiLCJ1cGRhdGVEaXNrU3RvcmFnZSIsImNhbGxlZFR3aWNlIiwic3RvcFVwZGF0ZUpvYiIsImNhbGxlZFRocmljZSIsIndyaXRlRmlsZVN5bmMiLCJoaXN0b3J5IiwiZ2V0SGlzdG9yeUZyb21EaXNrIiwicmVzIiwic2V0VGltZW91dCIsInNob3VsZCIsIm1hdGNoIiwic2V0U3RhcnROZXdPcmRlckluZGV4Iiwic2V0U3RhcnROZXdEZWFsSW5kZXgiLCJzYXZlZERhdGEiLCJhbGwiLCJyZWFkRmlsZSIsImVuc3VyZUZpbGUiLCJlcXVhbCIsImRlbGV0ZVN0b3JhZ2VGcm9tRGlzayJdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7Ozs7Ozs7QUFFQTs7OztBQUNBOzs7O0FBQ0E7Ozs7OztBQUNBLE1BQU1BLHFCQUFxQkMsUUFBUSxlQUFSLENBQTNCOztBQUVBOzs7QUFHQSxlQUFlQyxzQkFBZixHQUF3QztBQUN0QyxRQUFNQyxVQUFVO0FBQ2RDLFdBQU8sRUFETztBQUVkQyxtQkFBZSxFQUZEO0FBR2RDLGlDQUE2QixFQUhmO0FBSWRDLHlDQUFxQztBQUp2QixHQUFoQjtBQU1BLFFBQU1DLGlCQUFpQkMsa0JBQUdDLGNBQUgsQ0FBa0IsNkNBQWxCLENBQXZCO0FBQ0EsTUFBR0YsY0FBSCxFQUFtQjtBQUNqQixRQUFJRyxTQUFTQyxLQUFLQyxLQUFMLENBQVdKLGtCQUFHSyxZQUFILENBQWdCLDZDQUFoQixFQUErRCxPQUEvRCxFQUF3RUMsUUFBeEUsQ0FBaUYsT0FBakYsQ0FBWCxDQUFiO0FBQ0FaLFlBQVFHLDJCQUFSLEdBQXNDSyxPQUFPTCwyQkFBN0M7QUFDQUgsWUFBUUksbUNBQVIsR0FBOENJLE9BQU9KLG1DQUFyRDtBQUNEO0FBQ0QsUUFBTVMsZ0JBQWdCUCxrQkFBR0MsY0FBSCxDQUFrQiw0Q0FBbEIsQ0FBdEI7QUFDQSxNQUFHTSxhQUFILEVBQWtCO0FBQ2hCLFFBQUlaLFFBQVFRLEtBQUtDLEtBQUwsQ0FBV0osa0JBQUdLLFlBQUgsQ0FBZ0IsNENBQWhCLEVBQThELE9BQTlELEVBQXVFQyxRQUF2RSxDQUFnRixPQUFoRixDQUFYLENBQVo7QUFDQSxRQUFHWCxNQUFNYSxNQUFULEVBQWdCO0FBQ2RkLGNBQVFDLEtBQVIsR0FBZ0JBLE1BQU1jLEdBQU4sQ0FBV0MsSUFBRCxJQUFVO0FBQ2xDQSxhQUFLQyxJQUFMLEdBQVksSUFBSUMsSUFBSixDQUFTRixLQUFLQyxJQUFkLENBQVo7QUFDQSxlQUFPRCxJQUFQO0FBQ0QsT0FIZSxDQUFoQjtBQUlEO0FBQ0Y7QUFDRCxRQUFNRyxpQkFBaUJiLGtCQUFHQyxjQUFILENBQWtCLG9EQUFsQixDQUF2QjtBQUNBLE1BQUdZLGNBQUgsRUFBbUI7QUFDakIsUUFBSWpCLGdCQUFnQk8sS0FBS0MsS0FBTCxDQUFXSixrQkFBR0ssWUFBSCxDQUFnQixvREFBaEIsRUFDN0IsT0FENkIsRUFDcEJDLFFBRG9CLENBQ1gsT0FEVyxDQUFYLENBQXBCO0FBRUEsUUFBR1YsY0FBY1ksTUFBakIsRUFBeUI7QUFDdkJkLGNBQVFFLGFBQVIsR0FBd0JBLGNBQWNhLEdBQWQsQ0FBbUJLLFlBQUQsSUFBa0I7QUFDMURBLHFCQUFhSCxJQUFiLEdBQW9CLElBQUlDLElBQUosQ0FBU0UsYUFBYUgsSUFBdEIsQ0FBcEI7QUFDQUcscUJBQWFDLFFBQWIsR0FBd0IsSUFBSUgsSUFBSixDQUFTRSxhQUFhQyxRQUF0QixDQUF4QjtBQUNBLGVBQU9ELFlBQVA7QUFDRCxPQUp1QixDQUF4QjtBQUtEO0FBQ0Y7QUFDRCxTQUFPcEIsT0FBUDtBQUNEOztBQUVEOzs7QUFHQXNCLFNBQVMsb0JBQVQsRUFBK0IsTUFBTTs7QUFFbkMsTUFBSXRCLE9BQUo7QUFDQSxNQUFJdUIsV0FBSjtBQUNBLE1BQUlDLFFBQUo7QUFDQSxNQUFJQyxTQUFKO0FBQ0EsTUFBSUMsU0FBSjtBQUNBLE1BQUlDLFNBQUo7QUFDQSxNQUFJQyxVQUFKO0FBQ0EsTUFBSUMsVUFBSjtBQUNBLE1BQUlDLFVBQUo7QUFDQSxNQUFJQyxPQUFKOztBQUVBQyxTQUFPLE1BQU07QUFDWEQsY0FBVUUsZ0JBQU1DLGFBQU4sRUFBVjtBQUNBNUIsc0JBQUc2QixTQUFILENBQWEsWUFBYjtBQUNELEdBSEQ7O0FBS0FDLGFBQVcsWUFBWTtBQUNyQnBDLGNBQVUsRUFBVjtBQUNBdUIsa0JBQWMsSUFBSTFCLGtCQUFKLENBQXVCLFdBQXZCLEVBQW9DLGFBQXBDLEVBQW1ERyxPQUFuRCxDQUFkO0FBQ0ErQixZQUFRTSxJQUFSLENBQWFkLFdBQWIsRUFBMEIsZ0JBQTFCLEVBQTRDZSxPQUE1QztBQUNBZCxlQUFXLEVBQUNlLElBQUcsVUFBSixFQUFnQkMsTUFBSyxtQkFBckIsRUFBMENDLE9BQU0sQ0FBaEQsRUFBbUR4QixNQUFNLElBQUlDLElBQUosQ0FBUyxHQUFULENBQXpELEVBQXdFd0IsWUFBVyxDQUFuRjtBQUNUQyxZQUFLLENBREksRUFDREMsUUFBTyxLQUROLEVBQ2FDLFVBQVMsS0FEdEIsRUFDNkJDLFNBQVEsZ0JBRHJDLEVBQVg7QUFFQXJCLGdCQUFZLEVBQUNjLElBQUcsVUFBSixFQUFnQkMsTUFBSyxnQkFBckIsRUFBdUNDLE9BQU0sQ0FBN0MsRUFBZ0R4QixNQUFNLElBQUlDLElBQUosQ0FBUyxHQUFULENBQXRELEVBQXFFd0IsWUFBVyxDQUFoRjtBQUNWQyxZQUFLLENBREssRUFDRkMsUUFBTyxLQURMLEVBQ1lDLFVBQVMsS0FEckIsRUFDNEJDLFNBQVEsZ0JBRHBDLEVBQVo7QUFFQXBCLGdCQUFZLEVBQUNhLElBQUcsVUFBSixFQUFnQkMsTUFBSyxlQUFyQixFQUFzQ0MsT0FBTSxDQUE1QyxFQUErQ3hCLE1BQU0sSUFBSUMsSUFBSixDQUFTLEdBQVQsQ0FBckQsRUFBb0V3QixZQUFXLENBQS9FO0FBQ1ZDLFlBQUssQ0FESyxFQUNGQyxRQUFPLEtBREwsRUFDWUMsVUFBUyxLQURyQixFQUM0QkMsU0FBUSxnQkFEcEMsRUFBWjtBQUVBbkIsZ0JBQVksRUFBQ1ksSUFBRyxVQUFKLEVBQWdCQyxNQUFLLGlCQUFyQixFQUF3Q08sT0FBTSxvQkFBOUMsRUFBb0VDLFFBQU8sUUFBM0UsRUFBcUZQLE9BQU0sQ0FBM0Y7QUFDVnhCLFlBQU0sSUFBSUMsSUFBSixDQUFTLEVBQVQsQ0FESSxFQUNVRyxVQUFVLElBQUlILElBQUosQ0FBUyxHQUFULENBRHBCLEVBQ21DK0IsY0FBYSxDQURoRCxFQUNtREMsUUFBTyxJQUQxRDtBQUVWQyxxQkFBYyxDQUZKLEVBRU9DLFlBQVcsVUFGbEIsRUFFOEJQLFVBQVMsS0FGdkMsRUFFOENDLFNBQVEsdUJBRnRELEVBQVo7QUFHQWxCLGlCQUFhLEVBQUNXLElBQUcsVUFBSixFQUFnQkMsTUFBSyxzQkFBckIsRUFBNkNPLE9BQU0sb0JBQW5ELEVBQXlFQyxRQUFPLFFBQWhGLEVBQTBGUCxPQUFNLENBQWhHO0FBQ1h4QixZQUFNLElBQUlDLElBQUosQ0FBUyxFQUFULENBREssRUFDU0csVUFBVSxJQUFJSCxJQUFKLENBQVMsR0FBVCxDQURuQixFQUNrQytCLGNBQWEsQ0FEL0MsRUFDa0RDLFFBQU8sSUFEekQ7QUFFWEMscUJBQWMsQ0FGSCxFQUVNQyxZQUFXLFVBRmpCLEVBRTZCUCxVQUFTLEtBRnRDLEVBRTZDQyxTQUFRLHVCQUZyRCxFQUFiO0FBR0FqQixpQkFBYSxFQUFDVSxJQUFHLFVBQUosRUFBZ0JDLE1BQUssZ0JBQXJCLEVBQXVDTyxPQUFNLG9CQUE3QyxFQUFtRUMsUUFBTyxRQUExRSxFQUFvRlAsT0FBTSxDQUExRjtBQUNYeEIsWUFBTSxJQUFJQyxJQUFKLENBQVMsR0FBVCxDQURLLEVBQ1VHLFVBQVUsSUFBSUgsSUFBSixDQUFTLEdBQVQsQ0FEcEIsRUFDbUMrQixjQUFhLENBRGhELEVBQ21EQyxRQUFPLElBRDFEO0FBRVhDLHFCQUFjLENBRkgsRUFFTUMsWUFBVyxVQUZqQixFQUU2QlAsVUFBUyxLQUZ0QyxFQUU2Q0MsU0FBUSx1QkFGckQsRUFBYjtBQUdBaEIsaUJBQWE7QUFDWDNCLG1DQUE2QixFQUFDLEtBQUssYUFBTixFQURsQjtBQUVYQywyQ0FBcUMsRUFBQyxLQUFLLGFBQU47QUFGMUIsS0FBYjtBQUlELEdBdkJEOztBQXlCQWlELFFBQU0sTUFBTTtBQUNWL0Msc0JBQUdnRCxVQUFILENBQWMsWUFBZDtBQUNELEdBRkQ7O0FBSUFDLFlBQVUsTUFBTTtBQUNkeEIsWUFBUXlCLE9BQVI7QUFDQWxELHNCQUFHZ0QsVUFBSCxDQUFjLDRDQUFkO0FBQ0FoRCxzQkFBR2dELFVBQUgsQ0FBYyxvREFBZDtBQUNELEdBSkQ7O0FBTUE7Ozs7QUFJQUcsS0FBRywyQkFBSCxFQUFnQyxZQUFZO0FBQzFDMUIsWUFBUXlCLE9BQVI7QUFDQSxVQUFNRSxRQUFRekIsZ0JBQU0wQixhQUFOLEVBQWQ7QUFDQTVCLFlBQVFNLElBQVIsQ0FBYWQsV0FBYixFQUEwQixtQkFBMUIsRUFBK0NlLE9BQS9DO0FBQ0FmLGdCQUFZcUMsY0FBWjtBQUNBLFVBQU1GLE1BQU1HLFNBQU4sQ0FBZ0IsS0FBaEIsQ0FBTjtBQUNBNUIsb0JBQU02QixNQUFOLENBQWFDLFVBQWIsQ0FBd0J4QyxZQUFZeUMsaUJBQXBDO0FBQ0EsVUFBTU4sTUFBTUcsU0FBTixDQUFnQixLQUFoQixDQUFOO0FBQ0E1QixvQkFBTTZCLE1BQU4sQ0FBYUcsV0FBYixDQUF5QjFDLFlBQVl5QyxpQkFBckM7QUFDQXpDLGdCQUFZMkMsYUFBWjtBQUNBLFVBQU1SLE1BQU1HLFNBQU4sQ0FBZ0IsS0FBaEIsQ0FBTjtBQUNBNUIsb0JBQU02QixNQUFOLENBQWFHLFdBQWIsQ0FBeUIxQyxZQUFZeUMsaUJBQXJDO0FBQ0F6QyxnQkFBWXFDLGNBQVo7QUFDQSxVQUFNRixNQUFNRyxTQUFOLENBQWdCLEtBQWhCLENBQU47QUFDQTVCLG9CQUFNNkIsTUFBTixDQUFhSyxZQUFiLENBQTBCNUMsWUFBWXlDLGlCQUF0QztBQUNBekMsZ0JBQVkyQyxhQUFaO0FBQ0FSLFVBQU1GLE9BQU47QUFDRCxHQWpCRDs7QUFtQkE7OztBQUdBQyxLQUFHLCtCQUFILEVBQW9DLFlBQVk7QUFDOUNuRCxzQkFBRzhELGFBQUgsQ0FBaUIsNENBQWpCLEVBQStELHlCQUFlLENBQUM1QyxRQUFELENBQWYsQ0FBL0QsRUFDRSxPQURGO0FBRUFsQixzQkFBRzhELGFBQUgsQ0FBaUIsb0RBQWpCLEVBQXVFLHlCQUFlLENBQUN6QyxTQUFELENBQWYsQ0FBdkUsRUFDRSxPQURGO0FBRUFyQixzQkFBRzhELGFBQUgsQ0FBaUIsNkNBQWpCLEVBQWdFLHlCQUFldEMsVUFBZixDQUFoRSxFQUNFLE9BREY7QUFFQSxVQUFNdUMsVUFBVSxNQUFNOUMsWUFBWStDLGtCQUFaLEVBQXRCO0FBQ0EsVUFBTSxzQkFBWUMsT0FBT0MsV0FBV0QsR0FBWCxFQUFnQixFQUFoQixDQUFuQixDQUFOO0FBQ0FGLFlBQVFwRSxLQUFSLENBQWN3RSxNQUFkLENBQXFCQyxLQUFyQixDQUEyQixDQUFDbEQsUUFBRCxDQUEzQjtBQUNBNkMsWUFBUW5FLGFBQVIsQ0FBc0J1RSxNQUF0QixDQUE2QkMsS0FBN0IsQ0FBbUMsQ0FBQy9DLFNBQUQsQ0FBbkM7QUFDQTBDLFlBQVFsRSwyQkFBUixDQUFvQ3NFLE1BQXBDLENBQTJDQyxLQUEzQyxDQUFpRDVDLFdBQVczQiwyQkFBNUQ7QUFDQWtFLFlBQVFqRSxtQ0FBUixDQUE0Q3FFLE1BQTVDLENBQW1EQyxLQUFuRCxDQUF5RDVDLFdBQVcxQixtQ0FBcEU7QUFDRCxHQWJEOztBQWVBOzs7QUFHQXFELEtBQUcsNkJBQUgsRUFBa0MsWUFBWTtBQUM1Q3pELFlBQVFDLEtBQVIsR0FBZ0IsQ0FBQ3VCLFFBQUQsQ0FBaEI7QUFDQXhCLFlBQVFFLGFBQVIsR0FBd0IsQ0FBQ3lCLFNBQUQsQ0FBeEI7QUFDQTNCLFlBQVFHLDJCQUFSLEdBQXNDLEVBQUMsS0FBSyxhQUFOLEVBQXRDO0FBQ0FILFlBQVFJLG1DQUFSLEdBQThDLEVBQUMsS0FBSyxhQUFOLEVBQTlDO0FBQ0FtQixnQkFBWW9ELHFCQUFaLENBQWtDLENBQWxDO0FBQ0FwRCxnQkFBWXFELG9CQUFaLENBQWlDLENBQWpDO0FBQ0FyRCxnQkFBWXlDLGlCQUFaO0FBQ0EsVUFBTSxzQkFBWU8sT0FBT0MsV0FBV0QsR0FBWCxFQUFnQixFQUFoQixDQUFuQixDQUFOO0FBQ0EsVUFBTU0sWUFBWSxNQUFNOUUsd0JBQXhCO0FBQ0E4RSxjQUFVNUUsS0FBVixDQUFnQndFLE1BQWhCLENBQXVCQyxLQUF2QixDQUE2QixDQUFDbEQsUUFBRCxDQUE3QjtBQUNBcUQsY0FBVTNFLGFBQVYsQ0FBd0J1RSxNQUF4QixDQUErQkMsS0FBL0IsQ0FBcUMsQ0FBQy9DLFNBQUQsQ0FBckM7QUFDQWtELGNBQVUxRSwyQkFBVixDQUFzQ3NFLE1BQXRDLENBQTZDQyxLQUE3QyxDQUFtRCxFQUFDLEtBQUssYUFBTixFQUFuRDtBQUNBRyxjQUFVekUsbUNBQVYsQ0FBOENxRSxNQUE5QyxDQUFxREMsS0FBckQsQ0FBMkQsRUFBQyxLQUFLLGFBQU4sRUFBM0Q7QUFDRCxHQWREOztBQWdCQTs7O0FBR0FqQixLQUFHLG1DQUFILEVBQXdDLFlBQVk7QUFDbER6RCxZQUFRQyxLQUFSLEdBQWdCLENBQUN1QixRQUFELEVBQVdDLFNBQVgsQ0FBaEI7QUFDQXpCLFlBQVFFLGFBQVIsR0FBd0IsQ0FBQ3lCLFNBQUQsRUFBWUMsVUFBWixDQUF4QjtBQUNBTCxnQkFBWW9ELHFCQUFaLENBQWtDLENBQWxDO0FBQ0FwRCxnQkFBWXFELG9CQUFaLENBQWlDLENBQWpDO0FBQ0FyRCxnQkFBWXlDLGlCQUFaO0FBQ0EsVUFBTSxzQkFBWU8sT0FBT0MsV0FBV0QsR0FBWCxFQUFnQixFQUFoQixDQUFuQixDQUFOO0FBQ0E5QyxjQUFVZ0IsS0FBVixHQUFrQixHQUFsQjtBQUNBYixlQUFXYSxLQUFYLEdBQW1CLEdBQW5CO0FBQ0FsQixnQkFBWW9ELHFCQUFaLENBQWtDLENBQWxDO0FBQ0FwRCxnQkFBWXFELG9CQUFaLENBQWlDLENBQWpDO0FBQ0FyRCxnQkFBWXlDLGlCQUFaO0FBQ0EsVUFBTSxzQkFBWU8sT0FBT0MsV0FBV0QsR0FBWCxFQUFnQixFQUFoQixDQUFuQixDQUFOO0FBQ0EsVUFBTU0sWUFBWSxNQUFNOUUsd0JBQXhCO0FBQ0E4RSxjQUFVNUUsS0FBVixDQUFnQndFLE1BQWhCLENBQXVCQyxLQUF2QixDQUE2QixDQUFDbEQsUUFBRCxFQUFXQyxTQUFYLENBQTdCO0FBQ0FvRCxjQUFVM0UsYUFBVixDQUF3QnVFLE1BQXhCLENBQStCQyxLQUEvQixDQUFxQyxDQUFDL0MsU0FBRCxFQUFZQyxVQUFaLENBQXJDO0FBQ0QsR0FoQkQ7O0FBa0JBOzs7QUFHQTZCLEtBQUcsb0NBQUgsRUFBeUMsWUFBWTtBQUNuRHpELFlBQVFDLEtBQVIsR0FBZ0IsQ0FBQ3VCLFFBQUQsRUFBV0MsU0FBWCxDQUFoQjtBQUNBekIsWUFBUUUsYUFBUixHQUF3QixDQUFDeUIsU0FBRCxFQUFZQyxVQUFaLENBQXhCO0FBQ0FMLGdCQUFZb0QscUJBQVosQ0FBa0MsQ0FBbEM7QUFDQXBELGdCQUFZcUQsb0JBQVosQ0FBaUMsQ0FBakM7QUFDQSxVQUFNckQsWUFBWXlDLGlCQUFaLEVBQU47QUFDQXhDLGFBQVNpQixLQUFULEdBQWlCLEdBQWpCO0FBQ0FoQixjQUFVZ0IsS0FBVixHQUFrQixHQUFsQjtBQUNBZCxjQUFVYyxLQUFWLEdBQWtCLEdBQWxCO0FBQ0FiLGVBQVdhLEtBQVgsR0FBbUIsR0FBbkI7QUFDQWxCLGdCQUFZb0QscUJBQVosQ0FBa0MsQ0FBbEM7QUFDQXBELGdCQUFZcUQsb0JBQVosQ0FBaUMsQ0FBakM7QUFDQSxVQUFNckQsWUFBWXlDLGlCQUFaLEVBQU47QUFDQSxVQUFNYSxZQUFZLE1BQU05RSx3QkFBeEI7QUFDQThFLGNBQVU1RSxLQUFWLENBQWdCd0UsTUFBaEIsQ0FBdUJDLEtBQXZCLENBQTZCLENBQUNsRCxRQUFELEVBQVdDLFNBQVgsQ0FBN0I7QUFDQW9ELGNBQVUzRSxhQUFWLENBQXdCdUUsTUFBeEIsQ0FBK0JDLEtBQS9CLENBQXFDLENBQUMvQyxTQUFELEVBQVlDLFVBQVosQ0FBckM7QUFDRCxHQWhCRDs7QUFrQkE7OztBQUdBNkIsS0FBRyxrREFBSCxFQUF1RCxZQUFZO0FBQ2pFekQsWUFBUUMsS0FBUixHQUFnQixDQUFDdUIsUUFBRCxFQUFXQyxTQUFYLENBQWhCO0FBQ0F6QixZQUFRRSxhQUFSLEdBQXdCLENBQUN5QixTQUFELEVBQVlDLFVBQVosQ0FBeEI7QUFDQUwsZ0JBQVlvRCxxQkFBWixDQUFrQyxDQUFsQztBQUNBcEQsZ0JBQVlxRCxvQkFBWixDQUFpQyxDQUFqQztBQUNBLFVBQU1yRCxZQUFZeUMsaUJBQVosRUFBTjtBQUNBaEUsWUFBUUMsS0FBUixHQUFnQixDQUFDdUIsUUFBRCxFQUFXQyxTQUFYLEVBQXNCQyxTQUF0QixDQUFoQjtBQUNBMUIsWUFBUUUsYUFBUixHQUF3QixDQUFDeUIsU0FBRCxFQUFZQyxVQUFaLEVBQXdCQyxVQUF4QixDQUF4QjtBQUNBTixnQkFBWW9ELHFCQUFaLENBQWtDLENBQWxDO0FBQ0FwRCxnQkFBWXFELG9CQUFaLENBQWlDLENBQWpDO0FBQ0EsVUFBTXJELFlBQVl5QyxpQkFBWixFQUFOO0FBQ0EsVUFBTWEsWUFBWSxNQUFNOUUsd0JBQXhCO0FBQ0E4RSxjQUFVNUUsS0FBVixDQUFnQndFLE1BQWhCLENBQXVCQyxLQUF2QixDQUE2QixDQUFDbEQsUUFBRCxFQUFXQyxTQUFYLEVBQXNCQyxTQUF0QixDQUE3QjtBQUNBbUQsY0FBVTNFLGFBQVYsQ0FBd0J1RSxNQUF4QixDQUErQkMsS0FBL0IsQ0FBcUMsQ0FBQy9DLFNBQUQsRUFBWUMsVUFBWixFQUF3QkMsVUFBeEIsQ0FBckM7QUFDRCxHQWREOztBQWdCQTs7O0FBR0E0QixLQUFHLHFFQUFILEVBQTBFLFlBQVk7QUFDcEZ6RCxZQUFRQyxLQUFSLEdBQWdCLENBQUN1QixRQUFELEVBQVdDLFNBQVgsQ0FBaEI7QUFDQXpCLFlBQVFFLGFBQVIsR0FBd0IsQ0FBQ3lCLFNBQUQsRUFBWUMsVUFBWixDQUF4QjtBQUNBNUIsWUFBUUcsMkJBQVIsR0FBc0MsRUFBQyxLQUFLLGFBQU4sRUFBdEM7QUFDQUgsWUFBUUksbUNBQVIsR0FBOEMsRUFBQyxLQUFLLGFBQU4sRUFBOUM7QUFDQW1CLGdCQUFZb0QscUJBQVosQ0FBa0MsQ0FBbEM7QUFDQXBELGdCQUFZcUQsb0JBQVosQ0FBaUMsQ0FBakM7QUFDQSxVQUFNckQsWUFBWXlDLGlCQUFaLEVBQU47QUFDQWhFLFlBQVFDLEtBQVIsR0FBZ0IsQ0FBQ3VCLFFBQUQsRUFBV0MsU0FBWCxFQUFzQkMsU0FBdEIsQ0FBaEI7QUFDQTFCLFlBQVFFLGFBQVIsR0FBd0IsQ0FBQ3lCLFNBQUQsRUFBWUMsVUFBWixFQUF3QkMsVUFBeEIsQ0FBeEI7QUFDQTdCLFlBQVFHLDJCQUFSLEdBQXNDLEVBQUMsS0FBSyxhQUFOLEVBQXRDO0FBQ0FILFlBQVFJLG1DQUFSLEdBQThDLEVBQUMsS0FBSyxhQUFOLEVBQTlDO0FBQ0FtQixnQkFBWW9ELHFCQUFaLENBQWtDLENBQWxDO0FBQ0FwRCxnQkFBWXFELG9CQUFaLENBQWlDLENBQWpDO0FBQ0EsVUFBTSxrQkFBUUUsR0FBUixDQUFZLENBQ2hCdkQsWUFBWXlDLGlCQUFaLEVBRGdCLEVBRWhCekMsWUFBWXlDLGlCQUFaLEVBRmdCLEVBR2hCekMsWUFBWXlDLGlCQUFaLEVBSGdCLEVBSWhCekMsWUFBWXlDLGlCQUFaLEVBSmdCLEVBS2hCekMsWUFBWXlDLGlCQUFaLEVBTGdCLENBQVosQ0FBTjtBQU9BdkQsU0FBS0MsS0FBTCxFQUFXLE1BQU1KLGtCQUFHeUUsUUFBSCxDQUFZLG9EQUFaLENBQWpCO0FBQ0F0RSxTQUFLQyxLQUFMLEVBQVcsTUFBTUosa0JBQUd5RSxRQUFILENBQVksNENBQVosQ0FBakI7QUFDQXRFLFNBQUtDLEtBQUwsRUFBVyxNQUFNSixrQkFBR3lFLFFBQUgsQ0FBWSw2Q0FBWixDQUFqQjtBQUNELEdBeEJEOztBQTBCQTs7O0FBR0F0QixLQUFHLGlDQUFILEVBQXNDLFlBQVk7QUFDaEQsVUFBTW5ELGtCQUFHMEUsVUFBSCxDQUFjLG9EQUFkLENBQU47QUFDQSxVQUFNMUUsa0JBQUcwRSxVQUFILENBQWMsNENBQWQsQ0FBTjtBQUNBLFVBQU0xRSxrQkFBRzBFLFVBQUgsQ0FBYyw2Q0FBZCxDQUFOO0FBQ0ExRSxzQkFBR0MsY0FBSCxDQUFrQixvREFBbEIsRUFBd0VrRSxNQUF4RSxDQUErRVEsS0FBL0UsQ0FBcUYsSUFBckY7QUFDQTNFLHNCQUFHQyxjQUFILENBQWtCLDRDQUFsQixFQUFnRWtFLE1BQWhFLENBQXVFUSxLQUF2RSxDQUE2RSxJQUE3RTtBQUNBM0Usc0JBQUdDLGNBQUgsQ0FBa0IsNkNBQWxCLEVBQWlFa0UsTUFBakUsQ0FBd0VRLEtBQXhFLENBQThFLElBQTlFO0FBQ0EsVUFBTTFELFlBQVkyRCxxQkFBWixFQUFOO0FBQ0E1RSxzQkFBR0MsY0FBSCxDQUFrQixvREFBbEIsRUFBd0VrRSxNQUF4RSxDQUErRVEsS0FBL0UsQ0FBcUYsS0FBckY7QUFDQTNFLHNCQUFHQyxjQUFILENBQWtCLDRDQUFsQixFQUFnRWtFLE1BQWhFLENBQXVFUSxLQUF2RSxDQUE2RSxLQUE3RTtBQUNBM0Usc0JBQUdDLGNBQUgsQ0FBa0IsNkNBQWxCLEVBQWlFa0UsTUFBakUsQ0FBd0VRLEtBQXhFLENBQThFLEtBQTlFO0FBQ0QsR0FYRDtBQWFELENBM05EIiwiZmlsZSI6Im5vZGVNYW5hZ2VyLnNwZWMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbmltcG9ydCBzaG91bGQgZnJvbSAnc2hvdWxkJztcbmltcG9ydCBmcyBmcm9tICdmcy1leHRyYSc7XG5pbXBvcnQgc2lub24gZnJvbSAnc2lub24nO1xuY29uc3QgSGlzdG9yeUZpbGVNYW5hZ2VyID0gcmVxdWlyZSgnLi9ub2RlTWFuYWdlcicpO1xuXG4vKipcbiAqIEhlbHBlciBmdW5jdGlvbiB0byByZWFkIHNhdmVkIGhpc3Rvcnkgc3RvcmFnZVxuICovXG5hc3luYyBmdW5jdGlvbiByZWFkSGlzdG9yeVN0b3JhZ2VGaWxlKCkge1xuICBjb25zdCBzdG9yYWdlID0ge1xuICAgIGRlYWxzOiBbXSwgXG4gICAgaGlzdG9yeU9yZGVyczogW10sIFxuICAgIGxhc3REZWFsVGltZUJ5SW5zdGFuY2VJbmRleDoge30sXG4gICAgbGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXg6IHt9XG4gIH07XG4gIGNvbnN0IGlzQ29uZmlnRXhpc3RzID0gZnMucGF0aEV4aXN0c1N5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWNvbmZpZy5iaW4nKTtcbiAgaWYoaXNDb25maWdFeGlzdHMpIHtcbiAgICBsZXQgY29uZmlnID0gSlNPTi5wYXJzZShmcy5yZWFkRmlsZVN5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWNvbmZpZy5iaW4nLCAndXRmLTgnKS50b1N0cmluZygndXRmLTgnKSk7XG4gICAgc3RvcmFnZS5sYXN0RGVhbFRpbWVCeUluc3RhbmNlSW5kZXggPSBjb25maWcubGFzdERlYWxUaW1lQnlJbnN0YW5jZUluZGV4O1xuICAgIHN0b3JhZ2UubGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXggPSBjb25maWcubGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXg7XG4gIH1cbiAgY29uc3QgaXNEZWFsc0V4aXN0cyA9IGZzLnBhdGhFeGlzdHNTeW5jKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nKTtcbiAgaWYoaXNEZWFsc0V4aXN0cykge1xuICAgIGxldCBkZWFscyA9IEpTT04ucGFyc2UoZnMucmVhZEZpbGVTeW5jKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nLCAndXRmLTgnKS50b1N0cmluZygndXRmLTgnKSk7XG4gICAgaWYoZGVhbHMubGVuZ3RoKXtcbiAgICAgIHN0b3JhZ2UuZGVhbHMgPSBkZWFscy5tYXAoKGRlYWwpID0+IHtcbiAgICAgICAgZGVhbC50aW1lID0gbmV3IERhdGUoZGVhbC50aW1lKTtcbiAgICAgICAgcmV0dXJuIGRlYWw7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cbiAgY29uc3QgaXNPcmRlcnNFeGlzdHMgPSBmcy5wYXRoRXhpc3RzU3luYygnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24taGlzdG9yeU9yZGVycy5iaW4nKTtcbiAgaWYoaXNPcmRlcnNFeGlzdHMpIHtcbiAgICBsZXQgaGlzdG9yeU9yZGVycyA9IEpTT04ucGFyc2UoZnMucmVhZEZpbGVTeW5jKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1oaXN0b3J5T3JkZXJzLmJpbicsXG4gICAgICAndXRmLTgnKS50b1N0cmluZygndXRmLTgnKSk7XG4gICAgaWYoaGlzdG9yeU9yZGVycy5sZW5ndGgpIHtcbiAgICAgIHN0b3JhZ2UuaGlzdG9yeU9yZGVycyA9IGhpc3RvcnlPcmRlcnMubWFwKChoaXN0b3J5T3JkZXIpID0+IHtcbiAgICAgICAgaGlzdG9yeU9yZGVyLnRpbWUgPSBuZXcgRGF0ZShoaXN0b3J5T3JkZXIudGltZSk7XG4gICAgICAgIGhpc3RvcnlPcmRlci5kb25lVGltZSA9IG5ldyBEYXRlKGhpc3RvcnlPcmRlci5kb25lVGltZSk7XG4gICAgICAgIHJldHVybiBoaXN0b3J5T3JkZXI7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIHN0b3JhZ2U7XG59XG5cbi8qKlxuICogQHRlc3Qge0hpc3RvcnlGaWxlTWFuYWdlcn1cbiAqL1xuZGVzY3JpYmUoJ0hpc3RvcnlGaWxlTWFuYWdlcicsICgpID0+IHtcblxuICBsZXQgc3RvcmFnZTtcbiAgbGV0IGZpbGVNYW5hZ2VyO1xuICBsZXQgdGVzdERlYWw7XG4gIGxldCB0ZXN0RGVhbDI7XG4gIGxldCB0ZXN0RGVhbDM7XG4gIGxldCB0ZXN0T3JkZXI7XG4gIGxldCB0ZXN0T3JkZXIyO1xuICBsZXQgdGVzdE9yZGVyMztcbiAgbGV0IHRlc3RDb25maWc7XG4gIGxldCBzYW5kYm94O1xuXG4gIGJlZm9yZSgoKSA9PiB7XG4gICAgc2FuZGJveCA9IHNpbm9uLmNyZWF0ZVNhbmRib3goKTtcbiAgICBmcy5lbnN1cmVEaXIoJy4vLm1ldGFhcGknKTtcbiAgfSk7XG5cbiAgYmVmb3JlRWFjaChhc3luYyAoKSA9PiB7XG4gICAgc3RvcmFnZSA9IHt9O1xuICAgIGZpbGVNYW5hZ2VyID0gbmV3IEhpc3RvcnlGaWxlTWFuYWdlcignYWNjb3VudElkJywgJ2FwcGxpY2F0aW9uJywgc3RvcmFnZSk7XG4gICAgc2FuZGJveC5zdHViKGZpbGVNYW5hZ2VyLCAnc3RhcnRVcGRhdGVKb2InKS5yZXR1cm5zKCk7XG4gICAgdGVzdERlYWwgPSB7aWQ6JzM3ODYzNjQzJywgdHlwZTonREVBTF9UWVBFX0JBTEFOQ0UnLCBtYWdpYzowLCB0aW1lOiBuZXcgRGF0ZSgxMDApLCBjb21taXNzaW9uOjAsIFxuICAgICAgc3dhcDowLCBwcm9maXQ6MTAwMDAsIHBsYXRmb3JtOidtdDUnLCBjb21tZW50OidEZW1vIGRlcG9zaXQgMSd9O1xuICAgIHRlc3REZWFsMiA9IHtpZDonMzc4NjM2NDQnLCB0eXBlOidERUFMX1RZUEVfU0VMTCcsIG1hZ2ljOjEsIHRpbWU6IG5ldyBEYXRlKDIwMCksIGNvbW1pc3Npb246MCwgXG4gICAgICBzd2FwOjAsIHByb2ZpdDoxMDAwMCwgcGxhdGZvcm06J210NScsIGNvbW1lbnQ6J0RlbW8gZGVwb3NpdCAyJ307XG4gICAgdGVzdERlYWwzID0ge2lkOiczNzg2MzY0NScsIHR5cGU6J0RFQUxfVFlQRV9CVVknLCBtYWdpYzoyLCB0aW1lOiBuZXcgRGF0ZSgzMDApLCBjb21taXNzaW9uOjAsIFxuICAgICAgc3dhcDowLCBwcm9maXQ6MTAwMDAsIHBsYXRmb3JtOidtdDUnLCBjb21tZW50OidEZW1vIGRlcG9zaXQgMyd9O1xuICAgIHRlc3RPcmRlciA9IHtpZDonNjEyMTA0NjMnLCB0eXBlOidPUkRFUl9UWVBFX1NFTEwnLCBzdGF0ZTonT1JERVJfU1RBVEVfRklMTEVEJywgc3ltYm9sOidBVUROWkQnLCBtYWdpYzowLCBcbiAgICAgIHRpbWU6IG5ldyBEYXRlKDUwKSwgZG9uZVRpbWU6IG5ldyBEYXRlKDEwMCksIGN1cnJlbnRQcmljZToxLCB2b2x1bWU6MC4wMSwgXG4gICAgICBjdXJyZW50Vm9sdW1lOjAsIHBvc2l0aW9uSWQ6JzYxMjA2NjMwJywgcGxhdGZvcm06J210NScsIGNvbW1lbnQ6J0FTX0FVRE5aRF81WXlNNktTN0Z2Oid9O1xuICAgIHRlc3RPcmRlcjIgPSB7aWQ6JzYxMjEwNDY0JywgdHlwZTonT1JERVJfVFlQRV9CVVlfTElNSVQnLCBzdGF0ZTonT1JERVJfU1RBVEVfRklMTEVEJywgc3ltYm9sOidBVUROWkQnLCBtYWdpYzoxLCBcbiAgICAgIHRpbWU6IG5ldyBEYXRlKDc1KSwgZG9uZVRpbWU6IG5ldyBEYXRlKDIwMCksIGN1cnJlbnRQcmljZToxLCB2b2x1bWU6MC4wMSwgXG4gICAgICBjdXJyZW50Vm9sdW1lOjAsIHBvc2l0aW9uSWQ6JzYxMjA2NjMxJywgcGxhdGZvcm06J210NScsIGNvbW1lbnQ6J0FTX0FVRE5aRF81WXlNNktTN0Z2Oid9O1xuICAgIHRlc3RPcmRlcjMgPSB7aWQ6JzYxMjEwNDY1JywgdHlwZTonT1JERVJfVFlQRV9CVVknLCBzdGF0ZTonT1JERVJfU1RBVEVfRklMTEVEJywgc3ltYm9sOidBVUROWkQnLCBtYWdpYzoxLCBcbiAgICAgIHRpbWU6IG5ldyBEYXRlKDEwMCksIGRvbmVUaW1lOiBuZXcgRGF0ZSgzMDApLCBjdXJyZW50UHJpY2U6MSwgdm9sdW1lOjAuMDEsIFxuICAgICAgY3VycmVudFZvbHVtZTowLCBwb3NpdGlvbklkOic2MTIwNjYzMScsIHBsYXRmb3JtOidtdDUnLCBjb21tZW50OidBU19BVUROWkRfNVl5TTZLUzdGdjonfTtcbiAgICB0ZXN0Q29uZmlnID0ge1xuICAgICAgbGFzdERlYWxUaW1lQnlJbnN0YW5jZUluZGV4OiB7JzAnOiAxMDAwMDAwMDAwMDAwfSwgXG4gICAgICBsYXN0SGlzdG9yeU9yZGVyVGltZUJ5SW5zdGFuY2VJbmRleDogeycwJzogMTAwMDAwMDAwMDAxMH1cbiAgICB9O1xuICB9KTtcblxuICBhZnRlcigoKSA9PiB7XG4gICAgZnMucmVtb3ZlU3luYygnLi8ubWV0YWFwaScpO1xuICB9KTtcblxuICBhZnRlckVhY2goKCkgPT4ge1xuICAgIHNhbmRib3gucmVzdG9yZSgpO1xuICAgIGZzLnJlbW92ZVN5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWRlYWxzLmJpbicpO1xuICAgIGZzLnJlbW92ZVN5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWhpc3RvcnlPcmRlcnMuYmluJyk7XG4gIH0pO1xuICBcbiAgLyoqXG4gICAqIEB0ZXN0IHtIaXN0b3J5RmlsZU1hbmFnZXIjc3RhcnRVcGRhdGVKb2J9XG4gICAqIEB0ZXN0IHtIaXN0b3J5RmlsZU1hbmFnZXIjc3RvcFVwZGF0ZUpvYn1cbiAgICovXG4gIGl0KCdzaG91bGQgc3RhcnQgYW5kIHN0b3Agam9iJywgYXN5bmMgKCkgPT4ge1xuICAgIHNhbmRib3gucmVzdG9yZSgpO1xuICAgIGNvbnN0IGNsb2NrID0gc2lub24udXNlRmFrZVRpbWVycygpO1xuICAgIHNhbmRib3guc3R1YihmaWxlTWFuYWdlciwgJ3VwZGF0ZURpc2tTdG9yYWdlJykucmV0dXJucygpO1xuICAgIGZpbGVNYW5hZ2VyLnN0YXJ0VXBkYXRlSm9iKCk7XG4gICAgYXdhaXQgY2xvY2sudGlja0FzeW5jKDYxMDAwKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkT25jZShmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSk7XG4gICAgYXdhaXQgY2xvY2sudGlja0FzeW5jKDYxMDAwKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkVHdpY2UoZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UpO1xuICAgIGZpbGVNYW5hZ2VyLnN0b3BVcGRhdGVKb2IoKTtcbiAgICBhd2FpdCBjbG9jay50aWNrQXN5bmMoNjEwMDApO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRUd2ljZShmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSk7XG4gICAgZmlsZU1hbmFnZXIuc3RhcnRVcGRhdGVKb2IoKTtcbiAgICBhd2FpdCBjbG9jay50aWNrQXN5bmMoNjEwMDApO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRUaHJpY2UoZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UpO1xuICAgIGZpbGVNYW5hZ2VyLnN0b3BVcGRhdGVKb2IoKTtcbiAgICBjbG9jay5yZXN0b3JlKCk7XG4gIH0pO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7SGlzdG9yeUZpbGVNYW5hZ2VyI2dldEhpc3RvcnlGcm9tRGlza31cbiAgICovXG4gIGl0KCdzaG91bGQgcmVhZCBoaXN0b3J5IGZyb20gZmlsZScsIGFzeW5jICgpID0+IHtcbiAgICBmcy53cml0ZUZpbGVTeW5jKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nLCBKU09OLnN0cmluZ2lmeShbdGVzdERlYWxdKSxcbiAgICAgICd1dGYtOCcpO1xuICAgIGZzLndyaXRlRmlsZVN5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWhpc3RvcnlPcmRlcnMuYmluJywgSlNPTi5zdHJpbmdpZnkoW3Rlc3RPcmRlcl0pLFxuICAgICAgJ3V0Zi04Jyk7XG4gICAgZnMud3JpdGVGaWxlU3luYygnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24tY29uZmlnLmJpbicsIEpTT04uc3RyaW5naWZ5KHRlc3RDb25maWcpLFxuICAgICAgJ3V0Zi04Jyk7XG4gICAgY29uc3QgaGlzdG9yeSA9IGF3YWl0IGZpbGVNYW5hZ2VyLmdldEhpc3RvcnlGcm9tRGlzaygpO1xuICAgIGF3YWl0IG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgNTApKTtcbiAgICBoaXN0b3J5LmRlYWxzLnNob3VsZC5tYXRjaChbdGVzdERlYWxdKTtcbiAgICBoaXN0b3J5Lmhpc3RvcnlPcmRlcnMuc2hvdWxkLm1hdGNoKFt0ZXN0T3JkZXJdKTtcbiAgICBoaXN0b3J5Lmxhc3REZWFsVGltZUJ5SW5zdGFuY2VJbmRleC5zaG91bGQubWF0Y2godGVzdENvbmZpZy5sYXN0RGVhbFRpbWVCeUluc3RhbmNlSW5kZXgpO1xuICAgIGhpc3RvcnkubGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXguc2hvdWxkLm1hdGNoKHRlc3RDb25maWcubGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXgpO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge0hpc3RvcnlGaWxlTWFuYWdlciN1cGRhdGVEaXNrU3RvcmFnZX1cbiAgICovXG4gIGl0KCdzaG91bGQgc2F2ZSBpdGVtcyBpbiBhIGZpbGUnLCBhc3luYyAoKSA9PiB7XG4gICAgc3RvcmFnZS5kZWFscyA9IFt0ZXN0RGVhbF07XG4gICAgc3RvcmFnZS5oaXN0b3J5T3JkZXJzID0gW3Rlc3RPcmRlcl07XG4gICAgc3RvcmFnZS5sYXN0RGVhbFRpbWVCeUluc3RhbmNlSW5kZXggPSB7JzAnOiAxMDAwMDAwMDAwMDAwfTtcbiAgICBzdG9yYWdlLmxhc3RIaXN0b3J5T3JkZXJUaW1lQnlJbnN0YW5jZUluZGV4ID0geycwJzogMTAwMDAwMDAwMDAxMH07XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdPcmRlckluZGV4KDApO1xuICAgIGZpbGVNYW5hZ2VyLnNldFN0YXJ0TmV3RGVhbEluZGV4KDApO1xuICAgIGZpbGVNYW5hZ2VyLnVwZGF0ZURpc2tTdG9yYWdlKCk7XG4gICAgYXdhaXQgbmV3IFByb21pc2UocmVzID0+IHNldFRpbWVvdXQocmVzLCA1MCkpO1xuICAgIGNvbnN0IHNhdmVkRGF0YSA9IGF3YWl0IHJlYWRIaXN0b3J5U3RvcmFnZUZpbGUoKTtcbiAgICBzYXZlZERhdGEuZGVhbHMuc2hvdWxkLm1hdGNoKFt0ZXN0RGVhbF0pO1xuICAgIHNhdmVkRGF0YS5oaXN0b3J5T3JkZXJzLnNob3VsZC5tYXRjaChbdGVzdE9yZGVyXSk7XG4gICAgc2F2ZWREYXRhLmxhc3REZWFsVGltZUJ5SW5zdGFuY2VJbmRleC5zaG91bGQubWF0Y2goeycwJzogMTAwMDAwMDAwMDAwMH0pO1xuICAgIHNhdmVkRGF0YS5sYXN0SGlzdG9yeU9yZGVyVGltZUJ5SW5zdGFuY2VJbmRleC5zaG91bGQubWF0Y2goeycwJzogMTAwMDAwMDAwMDAxMH0pO1xuICB9KTtcblxuICAvKipcbiAgICogQHRlc3Qge0hpc3RvcnlGaWxlTWFuYWdlciN1cGRhdGVEaXNrU3RvcmFnZX1cbiAgICovXG4gIGl0KCdzaG91bGQgcmVwbGFjZSBOdGggaXRlbSBpbiBhIGZpbGUnLCBhc3luYyAoKSA9PiB7XG4gICAgc3RvcmFnZS5kZWFscyA9IFt0ZXN0RGVhbCwgdGVzdERlYWwyXTtcbiAgICBzdG9yYWdlLmhpc3RvcnlPcmRlcnMgPSBbdGVzdE9yZGVyLCB0ZXN0T3JkZXIyXTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld09yZGVySW5kZXgoMCk7XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdEZWFsSW5kZXgoMCk7XG4gICAgZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UoKTtcbiAgICBhd2FpdCBuZXcgUHJvbWlzZShyZXMgPT4gc2V0VGltZW91dChyZXMsIDUwKSk7XG4gICAgdGVzdERlYWwyLm1hZ2ljID0gMTAwO1xuICAgIHRlc3RPcmRlcjIubWFnaWMgPSAxMDA7XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdPcmRlckluZGV4KDEpO1xuICAgIGZpbGVNYW5hZ2VyLnNldFN0YXJ0TmV3RGVhbEluZGV4KDEpO1xuICAgIGZpbGVNYW5hZ2VyLnVwZGF0ZURpc2tTdG9yYWdlKCk7XG4gICAgYXdhaXQgbmV3IFByb21pc2UocmVzID0+IHNldFRpbWVvdXQocmVzLCA1MCkpO1xuICAgIGNvbnN0IHNhdmVkRGF0YSA9IGF3YWl0IHJlYWRIaXN0b3J5U3RvcmFnZUZpbGUoKTtcbiAgICBzYXZlZERhdGEuZGVhbHMuc2hvdWxkLm1hdGNoKFt0ZXN0RGVhbCwgdGVzdERlYWwyXSk7XG4gICAgc2F2ZWREYXRhLmhpc3RvcnlPcmRlcnMuc2hvdWxkLm1hdGNoKFt0ZXN0T3JkZXIsIHRlc3RPcmRlcjJdKTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtIaXN0b3J5RmlsZU1hbmFnZXIjdXBkYXRlRGlza1N0b3JhZ2V9XG4gICAqL1xuICBpdCgnc2hvdWxkIHJlcGxhY2UgYWxsIGl0ZW1zIGluIGEgZmlsZScsIGFzeW5jICgpID0+IHtcbiAgICBzdG9yYWdlLmRlYWxzID0gW3Rlc3REZWFsLCB0ZXN0RGVhbDJdO1xuICAgIHN0b3JhZ2UuaGlzdG9yeU9yZGVycyA9IFt0ZXN0T3JkZXIsIHRlc3RPcmRlcjJdO1xuICAgIGZpbGVNYW5hZ2VyLnNldFN0YXJ0TmV3T3JkZXJJbmRleCgwKTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld0RlYWxJbmRleCgwKTtcbiAgICBhd2FpdCBmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSgpO1xuICAgIHRlc3REZWFsLm1hZ2ljID0gMTAwO1xuICAgIHRlc3REZWFsMi5tYWdpYyA9IDEwMDtcbiAgICB0ZXN0T3JkZXIubWFnaWMgPSAxMDA7XG4gICAgdGVzdE9yZGVyMi5tYWdpYyA9IDEwMDtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld09yZGVySW5kZXgoMCk7XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdEZWFsSW5kZXgoMCk7XG4gICAgYXdhaXQgZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UoKTtcbiAgICBjb25zdCBzYXZlZERhdGEgPSBhd2FpdCByZWFkSGlzdG9yeVN0b3JhZ2VGaWxlKCk7XG4gICAgc2F2ZWREYXRhLmRlYWxzLnNob3VsZC5tYXRjaChbdGVzdERlYWwsIHRlc3REZWFsMl0pO1xuICAgIHNhdmVkRGF0YS5oaXN0b3J5T3JkZXJzLnNob3VsZC5tYXRjaChbdGVzdE9yZGVyLCB0ZXN0T3JkZXIyXSk7XG4gIH0pO1xuXG4gIC8qKlxuICAgKiBAdGVzdCB7SGlzdG9yeUZpbGVNYW5hZ2VyI3VwZGF0ZURpc2tTdG9yYWdlfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCBhcHBlbmQgYSBuZXcgb2JqZWN0IHRvIGFscmVhZHkgc2F2ZWQgb25lcycsIGFzeW5jICgpID0+IHtcbiAgICBzdG9yYWdlLmRlYWxzID0gW3Rlc3REZWFsLCB0ZXN0RGVhbDJdO1xuICAgIHN0b3JhZ2UuaGlzdG9yeU9yZGVycyA9IFt0ZXN0T3JkZXIsIHRlc3RPcmRlcjJdO1xuICAgIGZpbGVNYW5hZ2VyLnNldFN0YXJ0TmV3T3JkZXJJbmRleCgwKTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld0RlYWxJbmRleCgwKTtcbiAgICBhd2FpdCBmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSgpO1xuICAgIHN0b3JhZ2UuZGVhbHMgPSBbdGVzdERlYWwsIHRlc3REZWFsMiwgdGVzdERlYWwzXTtcbiAgICBzdG9yYWdlLmhpc3RvcnlPcmRlcnMgPSBbdGVzdE9yZGVyLCB0ZXN0T3JkZXIyLCB0ZXN0T3JkZXIzXTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld09yZGVySW5kZXgoMik7XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdEZWFsSW5kZXgoMik7XG4gICAgYXdhaXQgZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UoKTtcbiAgICBjb25zdCBzYXZlZERhdGEgPSBhd2FpdCByZWFkSGlzdG9yeVN0b3JhZ2VGaWxlKCk7XG4gICAgc2F2ZWREYXRhLmRlYWxzLnNob3VsZC5tYXRjaChbdGVzdERlYWwsIHRlc3REZWFsMiwgdGVzdERlYWwzXSk7XG4gICAgc2F2ZWREYXRhLmhpc3RvcnlPcmRlcnMuc2hvdWxkLm1hdGNoKFt0ZXN0T3JkZXIsIHRlc3RPcmRlcjIsIHRlc3RPcmRlcjNdKTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtIaXN0b3J5RmlsZU1hbmFnZXIjdXBkYXRlRGlza1N0b3JhZ2V9XG4gICAqL1xuICBpdCgnc2hvdWxkIG5vdCBjb3JydXB0IHRoZSBkaXNrIHN0b3JhZ2UgaWYgdXBkYXRlIGNhbGxlZCBtdWx0aXBsZSB0aW1lcycsIGFzeW5jICgpID0+IHtcbiAgICBzdG9yYWdlLmRlYWxzID0gW3Rlc3REZWFsLCB0ZXN0RGVhbDJdO1xuICAgIHN0b3JhZ2UuaGlzdG9yeU9yZGVycyA9IFt0ZXN0T3JkZXIsIHRlc3RPcmRlcjJdO1xuICAgIHN0b3JhZ2UubGFzdERlYWxUaW1lQnlJbnN0YW5jZUluZGV4ID0geycwJzogMTAwMDAwMDAwMDAwMH07XG4gICAgc3RvcmFnZS5sYXN0SGlzdG9yeU9yZGVyVGltZUJ5SW5zdGFuY2VJbmRleCA9IHsnMCc6IDEwMDAwMDAwMDAwMTB9O1xuICAgIGZpbGVNYW5hZ2VyLnNldFN0YXJ0TmV3T3JkZXJJbmRleCgwKTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld0RlYWxJbmRleCgwKTtcbiAgICBhd2FpdCBmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSgpO1xuICAgIHN0b3JhZ2UuZGVhbHMgPSBbdGVzdERlYWwsIHRlc3REZWFsMiwgdGVzdERlYWwzXTtcbiAgICBzdG9yYWdlLmhpc3RvcnlPcmRlcnMgPSBbdGVzdE9yZGVyLCB0ZXN0T3JkZXIyLCB0ZXN0T3JkZXIzXTtcbiAgICBzdG9yYWdlLmxhc3REZWFsVGltZUJ5SW5zdGFuY2VJbmRleCA9IHsnMSc6IDEwMDAwMDAwMDAwMDB9O1xuICAgIHN0b3JhZ2UubGFzdEhpc3RvcnlPcmRlclRpbWVCeUluc3RhbmNlSW5kZXggPSB7JzEnOiAxMDAwMDAwMDAwMDEwfTtcbiAgICBmaWxlTWFuYWdlci5zZXRTdGFydE5ld09yZGVySW5kZXgoMik7XG4gICAgZmlsZU1hbmFnZXIuc2V0U3RhcnROZXdEZWFsSW5kZXgoMik7XG4gICAgYXdhaXQgUHJvbWlzZS5hbGwoW1xuICAgICAgZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UoKSxcbiAgICAgIGZpbGVNYW5hZ2VyLnVwZGF0ZURpc2tTdG9yYWdlKCksXG4gICAgICBmaWxlTWFuYWdlci51cGRhdGVEaXNrU3RvcmFnZSgpLFxuICAgICAgZmlsZU1hbmFnZXIudXBkYXRlRGlza1N0b3JhZ2UoKSxcbiAgICAgIGZpbGVNYW5hZ2VyLnVwZGF0ZURpc2tTdG9yYWdlKClcbiAgICBdKTtcbiAgICBKU09OLnBhcnNlKGF3YWl0IGZzLnJlYWRGaWxlKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1oaXN0b3J5T3JkZXJzLmJpbicpKTtcbiAgICBKU09OLnBhcnNlKGF3YWl0IGZzLnJlYWRGaWxlKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nKSk7XG4gICAgSlNPTi5wYXJzZShhd2FpdCBmcy5yZWFkRmlsZSgnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24tY29uZmlnLmJpbicpKTtcbiAgfSk7XG5cbiAgLyoqXG4gICAqIEB0ZXN0IHtIaXN0b3J5RmlsZU1hbmFnZXIjZGVsZXRlU3RvcmFnZUZyb21EaXNrfVxuICAgKi9cbiAgaXQoJ3Nob3VsZCByZW1vdmUgaGlzdG9yeSBmcm9tIGRpc2snLCBhc3luYyAoKSA9PiB7XG4gICAgYXdhaXQgZnMuZW5zdXJlRmlsZSgnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24taGlzdG9yeU9yZGVycy5iaW4nKTtcbiAgICBhd2FpdCBmcy5lbnN1cmVGaWxlKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nKTtcbiAgICBhd2FpdCBmcy5lbnN1cmVGaWxlKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1jb25maWcuYmluJyk7XG4gICAgZnMucGF0aEV4aXN0c1N5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWhpc3RvcnlPcmRlcnMuYmluJykuc2hvdWxkLmVxdWFsKHRydWUpO1xuICAgIGZzLnBhdGhFeGlzdHNTeW5jKCcuLy5tZXRhYXBpL2FjY291bnRJZC1hcHBsaWNhdGlvbi1kZWFscy5iaW4nKS5zaG91bGQuZXF1YWwodHJ1ZSk7XG4gICAgZnMucGF0aEV4aXN0c1N5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWNvbmZpZy5iaW4nKS5zaG91bGQuZXF1YWwodHJ1ZSk7XG4gICAgYXdhaXQgZmlsZU1hbmFnZXIuZGVsZXRlU3RvcmFnZUZyb21EaXNrKCk7XG4gICAgZnMucGF0aEV4aXN0c1N5bmMoJy4vLm1ldGFhcGkvYWNjb3VudElkLWFwcGxpY2F0aW9uLWhpc3RvcnlPcmRlcnMuYmluJykuc2hvdWxkLmVxdWFsKGZhbHNlKTtcbiAgICBmcy5wYXRoRXhpc3RzU3luYygnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24tZGVhbHMuYmluJykuc2hvdWxkLmVxdWFsKGZhbHNlKTtcbiAgICBmcy5wYXRoRXhpc3RzU3luYygnLi8ubWV0YWFwaS9hY2NvdW50SWQtYXBwbGljYXRpb24tY29uZmlnLmJpbicpLnNob3VsZC5lcXVhbChmYWxzZSk7XG4gIH0pO1xuXG59KTtcbiJdfQ==