'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _entries = require('babel-runtime/core-js/object/entries');

var _entries2 = _interopRequireDefault(_entries);

var _latencyListener = require('../clients/metaApi/latencyListener');

var _latencyListener2 = _interopRequireDefault(_latencyListener);

var _statisticalReservoir = require('./reservoir/statisticalReservoir');

var _statisticalReservoir2 = _interopRequireDefault(_statisticalReservoir);

var _reservoir = require('./reservoir/reservoir');

var _reservoir2 = _interopRequireDefault(_reservoir);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Responsible for monitoring MetaApi application latencies
 */
class LatencyMonitor extends _latencyListener2.default {

  /**
   * Constructs latency monitor instance
   */
  constructor() {
    super();
    this._tradeReservoirs = {
      clientLatency: this._initializeReservoirs(),
      serverLatency: this._initializeReservoirs(),
      brokerLatency: this._initializeReservoirs()
    };
    this._updateReservoirs = {
      clientLatency: this._initializeReservoirs(),
      serverLatency: this._initializeReservoirs(),
      brokerLatency: this._initializeReservoirs()
    };
    this._priceReservoirs = {
      clientLatency: this._initializeReservoirs(),
      serverLatency: this._initializeReservoirs(),
      brokerLatency: this._initializeReservoirs()
    };
    this._requestReservoirs = {
      branch: true
    };
  }

  /**
   * Invoked with latency information when application receives a response to RPC request
   * @param {string} accountId account id
   * @param {string} type request type
   * @param {ResponseTimestamps} timestamps request timestamps object containing latency information
   */
  onResponse(accountId, type, timestamps) {
    if (!this._requestReservoirs[type]) {
      this._requestReservoirs[type] = {
        branch: true,
        clientLatency: this._initializeReservoirs(),
        serverLatency: this._initializeReservoirs()
      };
    }
    if (timestamps.serverProcessingStarted && timestamps.serverProcessingFinished) {
      let serverLatency = timestamps.serverProcessingFinished.getTime() - timestamps.serverProcessingStarted.getTime();
      this._saveMeasurement(this._requestReservoirs[type].serverLatency, serverLatency);
    }
    if (timestamps.clientProcessingStarted && timestamps.clientProcessingFinished && timestamps.serverProcessingStarted && timestamps.serverProcessingFinished) {
      let serverLatency = timestamps.serverProcessingFinished.getTime() - timestamps.serverProcessingStarted.getTime();
      let clientLatency = timestamps.clientProcessingFinished.getTime() - timestamps.clientProcessingStarted.getTime() - serverLatency;
      this._saveMeasurement(this._requestReservoirs[type].clientLatency, clientLatency);
    }
  }

  /**
   * Returns request processing latencies
   * @returns {Object} request processing latencies
   */
  get requestLatencies() {
    return this._constructLatenciesRecursively(this._requestReservoirs);
  }

  /**
   * Invoked with latency information when application receives symbol price update event
   * @param {string} accountId account id
   * @param {string} symbol price symbol
   * @param {SymbolPriceTimestamps} timestamps timestamps object containing latency information about price streaming
   */
  onSymbolPrice(accountId, symbol, timestamps) {
    if (timestamps.eventGenerated && timestamps.serverProcessingStarted) {
      let brokerLatency = timestamps.serverProcessingStarted.getTime() - timestamps.eventGenerated.getTime();
      this._saveMeasurement(this._priceReservoirs.brokerLatency, brokerLatency);
    }
    if (timestamps.serverProcessingStarted && timestamps.serverProcessingFinished) {
      let serverLatency = timestamps.serverProcessingFinished.getTime() - timestamps.serverProcessingStarted.getTime();
      this._saveMeasurement(this._priceReservoirs.serverLatency, serverLatency);
    }
    if (timestamps.serverProcessingFinished && timestamps.clientProcessingFinished) {
      let clientLatency = timestamps.clientProcessingFinished.getTime() - timestamps.serverProcessingFinished.getTime();
      this._saveMeasurement(this._priceReservoirs.clientLatency, clientLatency);
    }
  }

  /**
   * Returns price streaming latencies
   * @returns {Object} price streaming latencies
   */
  get priceLatencies() {
    return this._constructLatenciesRecursively(this._priceReservoirs);
  }

  /**
   * Invoked with latency information when application receives update event
   * @param {string} accountId account id
   * @param {UpdateTimestamps} timestamps timestamps object containing latency information about update streaming
   */
  onUpdate(accountId, timestamps) {
    if (timestamps.eventGenerated && timestamps.serverProcessingStarted) {
      let brokerLatency = timestamps.serverProcessingStarted.getTime() - timestamps.eventGenerated.getTime();
      this._saveMeasurement(this._updateReservoirs.brokerLatency, brokerLatency);
    }
    if (timestamps.serverProcessingStarted && timestamps.serverProcessingFinished) {
      let serverLatency = timestamps.serverProcessingFinished.getTime() - timestamps.serverProcessingStarted.getTime();
      this._saveMeasurement(this._updateReservoirs.serverLatency, serverLatency);
    }
    if (timestamps.serverProcessingFinished && timestamps.clientProcessingFinished) {
      let clientLatency = timestamps.clientProcessingFinished.getTime() - timestamps.serverProcessingFinished.getTime();
      this._saveMeasurement(this._updateReservoirs.clientLatency, clientLatency);
    }
  }

  /**
   * Returns update streaming latencies
   * @returns {Object} update streaming latencies
   */
  get updateLatencies() {
    return this._constructLatenciesRecursively(this._updateReservoirs);
  }

  /**
   * Invoked with latency information when application receives trade response
   * @param {string} accountId account id
   * @param {TradeTimestamps} timestamps timestamps object containing latency information about a trade
   */
  onTrade(accountId, timestamps) {
    if (timestamps.clientProcessingStarted && timestamps.serverProcessingStarted) {
      let clientLatency = timestamps.serverProcessingStarted.getTime() - timestamps.clientProcessingStarted.getTime();
      this._saveMeasurement(this._tradeReservoirs.clientLatency, clientLatency);
    }
    if (timestamps.serverProcessingStarted && timestamps.tradeStarted) {
      let serverLatency = timestamps.tradeStarted.getTime() - timestamps.serverProcessingStarted.getTime();
      this._saveMeasurement(this._tradeReservoirs.serverLatency, serverLatency);
    }
    if (timestamps.tradeStarted && timestamps.tradeExecuted) {
      let brokerLatency = timestamps.tradeExecuted.getTime() - timestamps.tradeStarted.getTime();
      this._saveMeasurement(this._tradeReservoirs.brokerLatency, brokerLatency);
    }
  }

  /**
   * Returns trade latencies
   * @returns {Object} trade latencies
   */
  get tradeLatencies() {
    return this._constructLatenciesRecursively(this._tradeReservoirs);
  }

  _saveMeasurement(reservoirs, clientLatency) {
    for (let e of (0, _entries2.default)(reservoirs)) {
      if (e[0] === 'branch') {
        continue;
      }
      e[1].percentiles.pushMeasurement(clientLatency);
      e[1].reservoir.pushMeasurement(clientLatency);
    }
  }

  _constructLatenciesRecursively(reservoirs) {
    let result = {};
    for (let e of (0, _entries2.default)(reservoirs)) {
      if (e[0] === 'branch') {
        continue;
      }
      result[e[0]] = e[1].branch ? this._constructLatenciesRecursively(e[1]) : {
        p50: e[1].percentiles.getPercentile(50),
        p75: e[1].percentiles.getPercentile(75),
        p90: e[1].percentiles.getPercentile(90),
        p95: e[1].percentiles.getPercentile(95),
        p98: e[1].percentiles.getPercentile(98),
        avg: e[1].reservoir.getStatistics().average,
        count: e[1].reservoir.getStatistics().count,
        min: e[1].reservoir.getStatistics().min,
        max: e[1].reservoir.getStatistics().max
      };
    }
    return result;
  }

  _initializeReservoirs() {
    return {
      branch: true,
      '1h': {
        percentiles: new _statisticalReservoir2.default(1000, 60 * 60 * 1000),
        reservoir: new _reservoir2.default(60, 60 * 60 * 1000)
      },
      '1d': {
        percentiles: new _statisticalReservoir2.default(1000, 24 * 60 * 60 * 1000),
        reservoir: new _reservoir2.default(60, 24 * 60 * 60 * 1000)
      },
      '1w': {
        percentiles: new _statisticalReservoir2.default(1000, 7 * 24 * 60 * 60 * 1000),
        reservoir: new _reservoir2.default(60, 7 * 24 * 60 * 60 * 1000)
      }
    };
  }

}
exports.default = LatencyMonitor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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