'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _historyStorage = require('./historyStorage');

var _historyStorage2 = _interopRequireDefault(_historyStorage);

var _index = require('./historyFileManager/index');

var _index2 = _interopRequireDefault(_index);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * History storage which stores MetaTrader history in RAM
 */
class MemoryHistoryStorage extends _historyStorage2.default {

  /**
   * Constructs the in-memory history store instance
   */
  constructor(accountId, application = 'MetaApi') {
    super();
    this._accountId = accountId;
    this._fileManager = new _index2.default(accountId, application, this);
    this._deals = [];
    this._historyOrders = [];
    this._lastDealTimeByInstanceIndex = {};
    this._lastHistoryOrderTimeByInstanceIndex = {};
    this._fileManager.startUpdateJob();
  }

  /**
   * Initializes the storage and loads required data from a persistent storage
   */
  async initialize() {
    await this.loadDataFromDisk();
  }

  /**
   * Returns all deals stored in history storage
   * @return {Array<MetatraderDeal>} all deals stored in history storage
   */
  get deals() {
    return this._deals;
  }

  /**
   * Returns all history orders stored in history storage
   * @return {Array<MetatraderOrder>} all history orders stored in history storage
   */
  get historyOrders() {
    return this._historyOrders;
  }

  /**
   * Returns times of last deals by instance indices
   * @return {Object} dictionary of last deal times by instance indices
   */
  get lastDealTimeByInstanceIndex() {
    return this._lastDealTimeByInstanceIndex;
  }

  /**
   * Returns times of last history orders by instance indices
   * @return {Object} dictionary of last history orders times by instance indices
   */
  get lastHistoryOrderTimeByInstanceIndex() {
    return this._lastHistoryOrderTimeByInstanceIndex;
  }

  /**
   * Resets the storage. Intended for use in tests
   */
  async clear() {
    this._deals = [];
    this._historyOrders = [];
    this._lastDealTimeByInstanceIndex = {};
    this._lastHistoryOrderTimeByInstanceIndex = {};
    await this._fileManager.deleteStorageFromDisk();
  }

  /**
   * Loads history data from the file manager
   * @return {Promise} promise which resolves when the history is loaded
   */
  async loadDataFromDisk() {
    const history = await this._fileManager.getHistoryFromDisk();
    this._deals = history.deals;
    this._historyOrders = history.historyOrders;
    this._lastDealTimeByInstanceIndex = history.lastDealTimeByInstanceIndex || {};
    this._lastHistoryOrderTimeByInstanceIndex = history.lastHistoryOrderTimeByInstanceIndex || {};
  }

  /**
   * Saves unsaved history items to disk storage
   */
  async updateDiskStorage() {
    await this._fileManager.updateDiskStorage();
  }

  /**
   * Returns the time of the last history order record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history order record stored in the history storage
   */
  lastHistoryOrderTime(instanceNumber) {
    if (instanceNumber !== undefined) {
      return new Date(this._lastHistoryOrderTimeByInstanceIndex['' + instanceNumber] || 0);
    } else {
      return new Date((0, _values2.default)(this._lastHistoryOrderTimeByInstanceIndex).reduce((acc, time) => time > acc ? time : acc, 0));
    }
  }

  /**
   * Returns the time of the last history deal record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history deal record stored in the history storage
   */
  lastDealTime(instanceNumber) {
    if (instanceNumber !== undefined) {
      return new Date(this._lastDealTimeByInstanceIndex['' + instanceNumber] || 0);
    } else {
      return new Date((0, _values2.default)(this._lastDealTimeByInstanceIndex).reduce((acc, time) => time > acc ? time : acc, 0));
    }
  }

  /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   */
  // eslint-disable-next-line complexity
  onHistoryOrderAdded(instanceIndex, historyOrder) {
    const instance = this.getInstanceNumber(instanceIndex);
    let insertIndex = 0;
    let replacementIndex = -1;
    const newHistoryOrderTime = (historyOrder.doneTime || new Date(0)).getTime();
    if (!this._lastHistoryOrderTimeByInstanceIndex['' + instance] || this._lastHistoryOrderTimeByInstanceIndex['' + instance] < newHistoryOrderTime) {
      this._lastHistoryOrderTimeByInstanceIndex['' + instance] = newHistoryOrderTime;
    }
    for (let i = this._historyOrders.length - 1; i >= 0; i--) {
      const order = this._historyOrders[i];
      const historyOrderTime = (order.doneTime || new Date(0)).getTime();
      if (historyOrderTime < newHistoryOrderTime || historyOrderTime === newHistoryOrderTime && order.id <= historyOrder.id) {
        if (historyOrderTime === newHistoryOrderTime && order.id === historyOrder.id && order.type === historyOrder.type) {
          replacementIndex = i;
        } else {
          insertIndex = i + 1;
        }
        break;
      }
    }
    if (replacementIndex !== -1) {
      this._historyOrders[replacementIndex] = historyOrder;
      this._fileManager.setStartNewOrderIndex(replacementIndex);
    } else {
      this._historyOrders.splice(insertIndex, 0, historyOrder);
      this._fileManager.setStartNewOrderIndex(insertIndex);
    }
  }

  /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   */
  // eslint-disable-next-line complexity
  onDealAdded(instanceIndex, deal) {
    const instance = this.getInstanceNumber(instanceIndex);
    let insertIndex = 0;
    let replacementIndex = -1;
    const newDealTime = (deal.time || new Date(0)).getTime();
    if (!this._lastDealTimeByInstanceIndex['' + instance] || this._lastDealTimeByInstanceIndex['' + instance] < newDealTime) {
      this._lastDealTimeByInstanceIndex['' + instance] = newDealTime;
    }
    for (let i = this._deals.length - 1; i >= 0; i--) {
      const d = this._deals[i];
      const dealTime = (d.time || new Date(0)).getTime();
      if (dealTime < newDealTime || dealTime === newDealTime && d.id <= deal.id || dealTime === newDealTime && d.id === deal.id && d.entryType <= deal.entryType) {
        if (dealTime === newDealTime && d.id === deal.id && d.entryType === deal.entryType) {
          replacementIndex = i;
        } else {
          insertIndex = i + 1;
        }
        break;
      }
    }
    if (replacementIndex !== -1) {
      this._deals[replacementIndex] = deal;
      this._fileManager.setStartNewDealIndex(replacementIndex);
    } else {
      this._deals.splice(insertIndex, 0, deal);
      this._fileManager.setStartNewDealIndex(insertIndex);
    }
  }

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized(instanceIndex, synchronizationId) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._dealSynchronizationFinished[instance] = true;
    await this.updateDiskStorage();
  }

}
exports.default = MemoryHistoryStorage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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