'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _provisioningProfileApi = require('./provisioningProfileApi');

var _provisioningProfileApi2 = _interopRequireDefault(_provisioningProfileApi);

var _provisioningProfile = require('./provisioningProfile');

var _provisioningProfile2 = _interopRequireDefault(_provisioningProfile);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {ProvisioningProfileApi}
 * @test {ProvisioningProfile}
 */
describe('ProvisioningProfileApi', () => {

  let sandbox;
  let api;
  let client = {
    getProvisioningProfiles: () => {},
    getProvisioningProfile: () => {},
    createProvisioningProfile: () => {},
    deleteProvisioningProfile: () => {},
    uploadProvisioningProfileFile: () => {},
    updateProvisioningProfile: () => {}
  };

  before(() => {
    api = new _provisioningProfileApi2.default(client);
    sandbox = _sinon2.default.createSandbox();
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {ProvisioningProfileApi#getProvisioningProfiles}
   */
  it('should retrieve provisioning profiles', async () => {
    sandbox.stub(client, 'getProvisioningProfiles').resolves([{ _id: 'id' }]);
    let profiles = await api.getProvisioningProfiles(4, 'new');
    profiles.map(p => p.id).should.match(['id']);
    profiles.forEach(p => (p instanceof _provisioningProfile2.default).should.be.true());
    _sinon2.default.assert.calledWith(client.getProvisioningProfiles, 4, 'new');
  });

  /**
   * @test {ProvisioningProfileApi#getProvisioningProfile}
   */
  it('should retrieve provisioning profile by id', async () => {
    sandbox.stub(client, 'getProvisioningProfile').resolves({
      _id: 'id', name: 'name', version: 4,
      status: 'new', brokerTimezone: 'EET', brokerDSTSwitchTimezone: 'EET'
    });
    let profile = await api.getProvisioningProfile('id');
    profile.id.should.equal('id');
    profile.name.should.equal('name');
    profile.version.should.equal(4);
    profile.status.should.equal('new');
    profile.brokerTimezone.should.equal('EET');
    profile.brokerDSTSwitchTimezone.should.equal('EET');
    (profile instanceof _provisioningProfile2.default).should.be.true();
    _sinon2.default.assert.calledWith(client.getProvisioningProfile, 'id');
  });

  /**
   * @test {ProvisioningProfileApi#createProvisioningProfiles}
   */
  it('should create provisioning profile', async () => {
    sandbox.stub(client, 'createProvisioningProfile').resolves({ id: 'id' });
    let profile = await api.createProvisioningProfile({
      name: 'name', version: 4,
      brokerTimezone: 'EET', brokerDSTSwitchTimezone: 'EET'
    });
    profile.id.should.equal('id');
    profile.name.should.equal('name');
    profile.version.should.equal(4);
    profile.status.should.equal('new');
    profile.brokerTimezone.should.equal('EET');
    profile.brokerDSTSwitchTimezone.should.equal('EET');
    (profile instanceof _provisioningProfile2.default).should.be.true();
    _sinon2.default.assert.calledWith(client.createProvisioningProfile, {
      name: 'name', version: 4,
      brokerTimezone: 'EET', brokerDSTSwitchTimezone: 'EET'
    });
  });

  /**
   * @test {ProvisioningProfile#reload}
   */
  it('should reload provisioning profile', async () => {
    sandbox.stub(client, 'getProvisioningProfile').onFirstCall().resolves({
      _id: 'id', name: 'name', version: 4, status: 'new',
      brokerTimezone: 'EET', brokerDSTSwitchTimezone: 'EET'
    }).onSecondCall().resolves({
      _id: 'id', name: 'name', version: 4, status: 'active',
      brokerTimezone: 'EET', brokerDSTSwitchTimezone: 'EET'
    });
    let profile = await api.getProvisioningProfile('id');
    await profile.reload();
    profile.status.should.equal('active');
    _sinon2.default.assert.calledWith(client.getProvisioningProfile, 'id');
    _sinon2.default.assert.calledTwice(client.getProvisioningProfile);
  });

  /**
   * @test {ProvisioningProfile#remove}
   */
  it('should remove provisioning profile', async () => {
    sandbox.stub(client, 'getProvisioningProfile').resolves({ _id: 'id', name: 'name', version: 4, status: 'new', brokerTimezone: 'EET',
      brokerDSTSwitchTimezone: 'EET' });
    sandbox.stub(client, 'deleteProvisioningProfile').resolves();
    let profile = await api.getProvisioningProfile('id');
    await profile.remove();
    _sinon2.default.assert.calledWith(client.deleteProvisioningProfile, 'id');
  });

  /**
   * @test {ProvisioningProfile#uploadFile}
   */
  it('should upload a file to provisioning profile', async () => {
    sandbox.stub(client, 'getProvisioningProfile').resolves({ _id: 'id', name: 'name', version: 4, status: 'new', brokerTimezone: 'EET',
      brokerDSTSwitchTimezone: 'EET' });
    sandbox.stub(client, 'uploadProvisioningProfileFile').resolves();
    let profile = await api.getProvisioningProfile('id');
    await profile.uploadFile('broker.srv', '/path/to/file.srv');
    _sinon2.default.assert.calledWith(client.uploadProvisioningProfileFile, 'id', 'broker.srv', '/path/to/file.srv');
  });

  /**
   * @test {ProvisioningProfile#update}
   */
  it('should update provisioning profile', async () => {
    sandbox.stub(client, 'getProvisioningProfile').resolves({ _id: 'id', name: 'name', version: 4, status: 'new', brokerTimezone: 'EET',
      brokerDSTSwitchTimezone: 'EET' });
    sandbox.stub(client, 'updateProvisioningProfile').resolves();
    let profile = await api.getProvisioningProfile('id');
    await profile.update({ name: 'name' });
    _sinon2.default.assert.calledWith(client.updateProvisioningProfile, 'id', { name: 'name' });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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