'use strict';

/**
 * FIFO-like reservoir of a fixed size capable
 * calculating running sums, min/max, average, msdev and stddev
 * msdev and stddev calculation by Naive algorithm
 * (Mean square deviation) msdev = sqrt((∑{i = from 1 to n}(Xi)^2 -(∑{i = from 1 to n}Xi)^2 / N) / N)
 * (Standard deviation) stddev = sqrt((∑{i = from 1 to n}(Xi)^2 -(∑{i = from 1 to n}Xi)^2 / N) / N - 1)
 * link: https://goo.gl/MAEGP2
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class Reservoir {

  /**
   * Constructs Reservoir
   * @param {number} size Reservoir size
   * @param {number} observationIntervalInMS Reservoir observation Interval In ms
   */
  constructor(size, observationIntervalInMS, object) {
    if (!object) {
      this.array = new Array();
      this.size = size;
      this._interval = observationIntervalInMS / size;
      this._queueEndTime = Date.now();
      this._fisrtQueueIndex = 0;
      this._intermediaryRecord = undefined;
      this.statistics = {
        count: 0,
        sum: 0,
        max: undefined,
        min: undefined,
        average: 0,
        sumOfSquares: 0,
        msdev: 0,
        stddev: 0
      };
    } else {
      this.array = object.array;
      this.size = object.size;
      this._interval = object._interval;
      this._queueEndTime = object._queueEndTime;
      this._fisrtQueueIndex = object._fisrtQueueIndex;
      this._intermediaryRecord = object._intermediaryRecord;
      this.statistics = this.checkStatisticsOnRestore(object.statistics);
    }
  }

  checkStatisticsOnRestore(statistics) {
    if (statistics.count === 0) {
      statistics = {
        count: 0,
        sum: 0,
        max: undefined,
        min: undefined,
        average: undefined,
        sumOfSquares: 0,
        msdev: undefined,
        stddev: undefined
      };
    } else if (statistics.count < 2) {
      statistics.msdev = undefined;
      statistics.stddev = undefined;
    }
    return statistics;
  }

  /**
   * Add element to Reservoir
   * @param {Number} data to add
   */
  pushMeasurement(data) {
    if (isFinite(data)) {
      this._updateQueue();
      this._updateIntermediaryRecord(data);
      this._updateStatisticsOnAdd(data);
    }
  }

  /**
   * return Reservoir statistics
   * @return {Object} Reservoir statistics
   */
  getStatistics() {
    this._updateQueue();
    return this.statistics;
  }

  toPlainObject() {
    this._updateQueue(true);
    return {
      array: this.array,
      size: this.size,
      _interval: this._interval,
      _queueEndTime: this._queueEndTime,
      _fisrtQueueIndex: this._fisrtQueueIndex,
      _intermediaryRecord: this._intermediaryRecord,
      statistics: this.statistics
    };
  }

  _updateQueue() {
    let intervalsCount = this._takeTimeIntervalsCount();
    let emptyElementsCount = this._takeEmptyElementsAddCount();
    if (emptyElementsCount > 0) {
      this._addRecord(emptyElementsCount);
      this._queueEndTime += intervalsCount * this._interval;
    }
  }

  _takeEmptyElementsAddCount() {
    let emptyElementsCount = this._takeTimeIntervalsCount();
    if (emptyElementsCount > this.size) {
      emptyElementsCount = this.size;
    }
    return emptyElementsCount;
  }

  _takeTimeIntervalsCount() {
    let timeNow = Date.now();
    let timeDiff = timeNow - this._queueEndTime;
    let timeIntervalsCount = Math.floor(timeDiff / this._interval);
    return timeIntervalsCount;
  }

  _updateRunningStatisticsOnRemove(removeCount) {
    let removeElementIndex = this._fisrtQueueIndex + 1;
    for (let i = 0; i < removeCount; i++) {
      if (removeElementIndex >= this.size) {
        removeElementIndex = 0;
      }

      this._updateStatisticsOnRemove(this.array[removeElementIndex], removeElementIndex);
      this.array[removeElementIndex] = {
        count: 0,
        sum: 0,
        max: undefined,
        min: undefined,
        average: 0,
        sumOfSquares: 0
      };
      removeElementIndex++;
    }
    removeElementIndex--;
    if (removeElementIndex < 0) {
      removeElementIndex = this.size - 1;
    }
    return removeElementIndex;
  }

  _updateStatisticsOnRemove(removeElement, removeElementIndex) {
    if (removeElement !== undefined && removeElement !== null) {
      this.statistics.count -= removeElement.count;
      this.statistics.sumOfSquares -= removeElement.sumOfSquares;
      this.statistics.sum -= removeElement.sum;
      this._updateStatisticsMinAndMaxOnRemove(removeElement, removeElementIndex);
      if (this.statistics.count > 0) {
        this.statistics.average = this.statistics.sum / this.statistics.count;
        if (this.statistics.count > 1) {
          let difOfSums = this._calculateDifferenceOfSums(this.statistics.sumOfSquares, this.statistics.sum, this.statistics.count);
          this.statistics.msdev = parseFloat(Math.sqrt(difOfSums / this.statistics.count));
          this.statistics.stddev = parseFloat(Math.sqrt(difOfSums / (this.statistics.count - 1)));
        } else {
          this.statistics.stddev = undefined;
          this.statistics.msdev = undefined;
        }
      } else {
        this.statistics.average = undefined;
        this.statistics.stddev = undefined;
        this.statistics.msdev = undefined;
      }
    }
  }

  _updateStatisticsMinAndMaxOnRemove(removeElement, removeElementIndex) {
    if (removeElement.max !== undefined && removeElement.max === this.statistics.max) {
      this.statistics.max = this._findMax(removeElementIndex);
    }

    if (removeElement.min !== undefined && removeElement.min === this.statistics.min) {
      this.statistics.min = this._findMin(removeElementIndex);
    }
  }

  _updateStatisticsOnAdd(el) {
    if (el !== undefined && el !== null) {
      this.statistics.count += 1;
      this.statistics.sum += el;
      this._updateStatisticsMinAndMaxOnAdd(el);
      this.statistics.sumOfSquares += Math.pow(el, 2);
      if (this.statistics.count > 0) {
        this.statistics.average = this.statistics.sum / this.statistics.count;
        let difOfSums = this._calculateDifferenceOfSums(this.statistics.sumOfSquares, this.statistics.sum, this.statistics.count);
        if (this.statistics.count > 1) {
          this.statistics.msdev = parseFloat(Math.sqrt(difOfSums / this.statistics.count));
          this.statistics.stddev = parseFloat(Math.sqrt(difOfSums / (this.statistics.count - 1)));
        } else {
          this.statistics.msdev = undefined;
          this.statistics.stddev = undefined;
        }
      }
    }
  }

  _updateStatisticsMinAndMaxOnAdd(el) {
    if (this.statistics.max < el || this.statistics.max === undefined || this.statistics.max === null) {
      this.statistics.max = el;
    }
    if (this.statistics.min > el || this.statistics.min === undefined || this.statistics.min === null) {
      this.statistics.min = el;
    }
  }

  _addRecord(emptyElementsCount) {
    if (this._intermediaryRecord !== undefined) {
      this.array[this._fisrtQueueIndex] = this._intermediaryRecord;
      this._intermediaryRecord = undefined;
    }
    let curIndexInArray = this._updateRunningStatisticsOnRemove(emptyElementsCount);
    this._fisrtQueueIndex = curIndexInArray;
  }

  _calculateDifferenceOfSums(sum1, sum2, count) {
    let dif = sum1 - Math.pow(sum2, 2) / count;
    return dif;
  }

  _updateIntermediaryRecord(el) {
    if (this._intermediaryRecord === undefined) {
      this._intermediaryRecord = {
        count: 1,
        sum: el,
        max: el,
        min: el,
        average: el,
        sumOfSquares: Math.pow(el, 2)
      };
    } else {
      if (this._intermediaryRecord.max < el) {
        this._intermediaryRecord.max = el;
      }
      if (this._intermediaryRecord.min > el) {
        this._intermediaryRecord.min = el;
      }
      this._intermediaryRecord.count += 1;
      this._intermediaryRecord.sum += el;
      this._intermediaryRecord.sumOfSquares += Math.pow(el, 2);
    }
  }

  _findMin(index) {
    let min = Infinity;
    this.array.forEach((el, i) => {
      if (el !== null && el !== undefined && el.min !== undefined && el.min < min && i !== index) {
        min = el.min;
      }
    });
    if (min === Infinity) {
      return this._intermediaryRecord !== undefined ? this._intermediaryRecord.min : undefined;
    }
    return min;
  }

  _findMax(index) {
    let max = -Infinity;

    this.array.forEach((el, i) => {
      if (el !== null && el !== undefined && el.max !== undefined && el.max > max && i !== index) {
        max = el.max;
      }
    });
    if (max === -Infinity) {
      return this._intermediaryRecord !== undefined ? this._intermediaryRecord.max : undefined;
    }
    return max;
  }
}
exports.default = Reservoir;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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