const FileManager = require('./fileManager');
const { openDB, deleteDB } = require('idb');
const LoggerManager = require('../../logger');

/**
 * History storage manager which saves and loads history to IndexedDB
 */
module.exports = class BrowserHistoryManager extends FileManager {

  /**
   * Constructs the history file manager instance
   * @param {String} accountId MetaApi account id
   * @param {String} application MetaApi application id
   * @param {HistoryStorage} historyStorage history storage
   */
  constructor(accountId, application, historyStorage) {
    super(accountId, application, historyStorage);
    this._logger = LoggerManager.getLogger('BrowserHistoryManager');
  }

  /**
   * Retrieves history from IndexedDB
   * @returns {Object} object with deals and historyOrders
   */
  // eslint-disable-next-line complexity
  async getHistoryFromDisk() {
    try {
      const history = {
        deals: [], 
        historyOrders: [], 
        lastDealTimeByInstanceIndex: {}, 
        lastHistoryOrderTimeByInstanceIndex: {}
      };
      const db = await this._getDatabase();
      const deals = await db.get('deals', this._accountId + '-' + this._application);
      history.deals = deals && deals.items || [];
      history.lastDealTimeByInstanceIndex = deals && deals.lastDealTimeByInstanceIndex || {};
      const historyOrders = await db.get('historyOrders', this._accountId + '-' + this._application);
      history.historyOrders = historyOrders && historyOrders.items || [];
      history.lastHistoryOrderTimeByInstanceIndex = historyOrders && 
        historyOrders.lastHistoryOrderTimeByInstanceIndex || {};
      db.close();
      return history;
    } catch(err) {
      this._logger.error(`${this._accountId}: Failed to get history from IndexedDB`, err);
    }
  }

  /**
   * Saves history items to IndexedDB
   */
  async updateDiskStorage() {
    try {
      const db = await this._getDatabase();
      await db.put('deals', {accountIdAndApplication: this._accountId + '-' + this._application,
        items: this._historyStorage.deals, 
        lastDealTimeByInstanceIndex: this._historyStorage.lastDealTimeByInstanceIndex});
      await db.put('historyOrders', {accountIdAndApplication: this._accountId + '-' + this._application,
        items: this._historyStorage.historyOrders,
        lastHistoryOrderTimeByInstanceIndex: this._historyStorage.lastHistoryOrderTimeByInstanceIndex});
      db.close();
    } catch(err) {
      this._logger.error(`${this._accountId}: Failed to save history into IndexedDB`, err);
    }
  }

  /**
   * Deletes storage files from disk
   */
  async deleteStorageFromDisk(){
    try {
      const db = await this._getDatabase();
      await db.delete('deals', this._accountId + '-' + this._application);
      await db.delete('historyOrders', this._accountId + '-' + this._application);
      db.close();
    } catch(err) {
      this._logger.error(`${this._accountId}: Failed to delete history from IndexedDB`, err);
    }
  }

  /**
   * Opens an IndexedDB database and verifies its structure, recreates if structure is invalid, then returns
   * @returns {IndexedDB} indexed db
   */
  async _getDatabase() {
    const keyPath = 'accountIdAndApplication';
    const db = await openDB('metaapi', 1, {
      upgrade(database, oldVersion, newVersion, transaction) {
        if (!database.objectStoreNames.contains('deals')) {
          database.createObjectStore('deals', {keyPath});
        }
        if (!database.objectStoreNames.contains('historyOrders')) {
          database.createObjectStore('historyOrders', {keyPath});
        }
      },
    });
    if(!db.objectStoreNames.contains('deals') || !db.objectStoreNames.contains('historyOrders')) {
      db.close();
      await deleteDB('metaapi');
      return await this._getDatabase();
    } else {
      const tr = db.transaction(['deals', 'historyOrders'], 'readonly');  
      if(tr.objectStore('deals').keyPath !== keyPath || tr.objectStore('historyOrders').keyPath !== keyPath) {
        db.close();
        await deleteDB('metaapi');
        return await this._getDatabase();
      } else { 
        return db;
      }
    }
  }

};